#-*- coding: utf-8 -*-
"""
Agent zephir pour la surveillance des drapeaux déclarés dans la configuration
"""

from twisted.internet.utils import getProcessOutput
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff, log
import json
import datetime

def schedule_to_img(schedule):
    if not schedule:
        return "N/A"
    table = """<table style="border:none;width:100%"><tr>{labels}</tr><tr>{segments}</tr></table>"""
    label = """<td style="padding:0;border:none"><table style="padding:0;border:none; width:100%"><tr>{times}</tr></table></td>"""
    item_label = """<td style="padding:0;border:none;text-align:left">{start_m}</td><td style="border:none;width:100%"></td><td style="padding:0;border:none;text-align:right">{end_h}</td>"""
    segment = """<td style="background-color:{color};height:20px;width:{width}%;border:none"></td>"""
    img = []
    hrs = []

    def width(delta):
        total_width = 100
        total_time = 24 * 3600
        width = int(delta.seconds * total_width / total_time)
        return width

    for index, event in enumerate(schedule):
        start_h, start_m = [t for t in event[0].split(":")]
        end_h, end_m = [t for t in event[1].split(":")]
        start = datetime.timedelta(hours=int(start_h), minutes=int(start_m))
        end = datetime.timedelta(hours=int(end_h), minutes=int(end_m))
        color = 'green' if event[2] == 'on' else 'red'
        img.append(segment.format(color=color, width=width(end-start)))
        if index == 0:
            left_label = "{}h{}".format(start_h, start_m)
            if len(schedule) == 1:
                right_label = "{}h{}".format(end_h, end_m)
            else:
                right_label = "{}h".format(end_h)
        elif index + 1 == len(schedule):
            left_label = start_m
            right_label = "{}h{}".format(end_h, end_m)
        else:
            left_label = start_m
            right_label = "{}h".format(end_h)
        hrs.append(label.format(times=item_label.format(start_m=left_label, end_h=right_label)))
    graph_table = table.format(segments="".join(img), labels="".join(hrs))
    return graph_table


class Flags(Agent):
    def __init__(self, name, flags_list, **args):
        Agent.__init__(self, name, **args)
        self.flags = flags_list
        self.table = TableData([
            ('description', 'Drapeau', {'align': 'left'}, None),
            ('status', "Etat", {'align':'center'}, status_to_img),
            ('mode', 'Actualisation', {'align': 'left'}, None),
            ('emplacement', "Emplacement", {'align':'left'}, None),
            ('planning', 'Planning du jour', {'align': 'center', 'width': '100%'}, None),
            ])
        self.data = [self.table]

    def measure(self):
        mesures = []
        for flag in self.flags:
            if flag in self.manager.agents and self.manager.agents[flag].last_measure is not None:
                serv_agent = self.manager.agents[flag]
                status = serv_agent.last_measure.value['flags'][0]['status']
                description = serv_agent.last_measure.value['flags'][0]['description']
                #status = boolean_to_onoff(1)
                #description = flag
                emplacement = serv_agent.last_measure.value['flags'][0]['emplacement']
                planning = serv_agent.last_measure.value['flags'][0]['planning']
                mode = serv_agent.last_measure.value['flags'][0]['mode']
                mesures.append({ 'description': description,
                            'status': status,
                            'emplacement': emplacement,
                            "planning": planning,
                            "mode": mode,
                            'stats': '<img src=/agents/status.png alt="Historique ()"/>'})
                self.measure_data[flag]=[status,description]
            else:
                mesures.append({ 'description': flag,
                            'status': '',
                            'emplacement': "",
                            "planning": "",
                            "mode": "",
                            'stats': ''})
        return {'flags': mesures}


    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['flags']


    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['flags']:
                if service['status'] == 'Off':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()

class Flag(Agent):

    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.name = name
        self.table = TableData([
            ('status', "Etat", {'align':'center'}, status_to_img),
            ('mode', 'Actualisation', {'align': 'left'}, None),
            ('emplacement', 'Emplacement', {'align': 'center'}, None),
            ('planning', 'Planning', {'align': 'center'}, None),
            ])
        self.data = [self.table]

    def measure(self):
        result = getProcessOutput("/usr/sbin/flag-switch",
                                  args=('-c', '/etc/flag-switch.conf', 'test', '-n', self.name),
                                  env={'LC_ALL': 'C'})
        result.addCallbacks(self.measure_activity, self.measure_error)
        return result

    def measure_activity(self, data):
        report = json.loads(data)

        status = boolean_to_onoff(report[u"applied"])
        mesures = [{'description': self.name,
                    'emplacement': report[u"location"],
                    "planning": schedule_to_img(report.get(u"planning", None)),
                    "mode": report[u"mode"],
                    'status': status }]
        return {'flags': mesures}

    def measure_error(self, failure):
        log.msg(u'Erreur remontée par /usr/sbin/flag-switch :')
        log.msg(failure.getErrorMessage())

        status = boolean_to_onoff(0)
        mesures = [{'description': self.name,
                    'status': status,
                    'emplacement': "",
                    "planning": "",
                    "mode": "",
                    }]
        return {'flags': mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['flags']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None and self.last_measure.value is not None:
            for service in self.last_measure.value['flags']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()
