#-*- coding: utf-8 -*-
from twisted.internet.utils import getProcessOutput
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff, log

class DHCPService(Agent):

    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.table = TableData([
            ('status', "État", {'align': 'center'}, status_to_img),
            ('name', "Plage", {'align': 'left'}, None),
            ('leases', "IP disponibles", {'align': 'right'}, None),
            ])
        self.data = [self.table]
        self.status = status.OK()

    def measure(self):
        DHCP_TOOL = '/usr/share/eole/sbin/dhcp-tool'
        result = getProcessOutput(DHCP_TOOL)
        result.addCallbacks(self.process_measure, self.measure_error)
        return result

    def process_measure(self, data):
        data = [d.split(b' ') for d in data.strip().split(b'\n')]
        l = []
        for d in data:
            ratio = float(d[1])/int(d[2])
            warning_ratio = float(d[3])/100
            error_if_full = True if d[4] == b'oui' else False
            if ratio == 0:
                if error_if_full:
                    status = boolean_to_onoff(0)
                else:
                    status = ''
            elif ratio <= warning_ratio:
                status = ''
            else:
                status = boolean_to_onoff(1)
            l.append({'status': status,
                      'name': d[0].decode(),
                      'leases': '{}/{}'.format(d[1].decode(), d[2].decode())})
        return {'leases': l}

    def measure_error(self, failure):
        log.msg(u'Erreur remontée par /usr/share/eole/sbin/dhcp-tool')
        log.msg(failure.getErrorMessage())

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None and self.last_measure.value is not None:
            self.table.table_data = self.last_measure.value['leases']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None and self.last_measure.value is not None:
            levels = [status.OK() if s['status'] != 'Off' else status.Error()
                      for s in self.last_measure.value['leases']]
            return max(levels)
        return status.Unknown()
