"""this module defines the list of exceptions used by CAS server
"""
from xmlrpclib import escape

class Redirect(Exception):
    def __init__(self, location):
        super(Exception, self).__init__(location)
        self.location = location

class CasServerException(Exception):
    """base class for all server exceptions"""
    
class InvalidTicketString(CasServerException):
    """raised when the application ticket does not match
    any generated application ticket for the given sessionid
    and service url.
    """

class CasError(CasServerException):
    def __init__(self, code, reason):
        # super(InvalidRequest, self).__init__(reason)
        self.reason = reason
        self.code = code

    def as_xml(self):
        return """<cas:ServiceResponse xmlns:cas="http://www.yale.edu/tp.cas">
  <cas:authenticationFailure code="%s">
    %s
  </cas:authenticationFailure>
</cas:ServiceResponse>
""" % (escape(self.code), escape(self.reason))

class InvalidRequest(CasError):
    def __init__(self, reason):
        super(InvalidRequest, self).__init__('INVALID_REQUEST', reason)

class MissingParameters(InvalidRequest):
    def __init__(self, parameters):
        msg = 'missing required parameters : %s' % ', '.join(parameters)
        super(MissingParameters, self).__init__(msg)
 
class InvalidTicket(CasError):
    def __init__(self, ticket, additional_info=''):
        msg = 'Ticket %s not recognized' % ticket
        if additional_info:
            msg += ' (%s)' % additional_info
        super(InvalidTicket, self).__init__('INVALID_TICKET', msg)
 
class InvalidService(CasError):
    def __init__(self, service, additional_info=''):
        msg = 'Service %s is not valid' % service
        if additional_info:
            msg += ' (%s)' % additional_info
        super(InvalidService, self).__init__('INVALID_SERVICE', msg)

class InternalError(CasError):
    def __init__(self, reason):
        super(InternalError, self).__init__('INTERNAL_ERROR', reason)
