#! /usr/bin/python3
# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2013
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
interroge la base AAF
pour generer des fichiers ldiff
"""
import sys
from os.path import join
# cheetah templating
from creole.template import CreoleTemplateEngine
from creole.client import CreoleClient
from Cheetah.Template import Template as CheetahTemplate

from eoleaaf.ldiftranslator import LdifTranslator, gen_group_cn, \
    gen_group_owner, gen_user_cn, gen_user_displayname
from eoleaaf.config import template_path, ldap_delta_create_filename, \
    ldap_delta_update_filename, ldap_delta_delete_filename, usersdn, \
    entservac_structures, dbtype, import_responsable
from eoleaaf.util import clean_empty_keys, modif_fields, \
    SimpleConnectionWrapper, log_error, make_datetime, log

if dbtype in ["mysql", "sqlite"]:
    from eoleaaf.util import MinimalSQLGenerator as DataBaseGenerator
elif dbtype == "mongodb":
    from eoleaaf.util import MongoDBGenerator as DataBaseGenerator

# ____________________________________________________________
# global (and unique) sql generator object
sqlgen = DataBaseGenerator()

client = CreoleClient()

# global (and unique) database connection
db = SimpleConnectionWrapper()
db.connect(default_cursor=False)

translator = LdifTranslator()

def ldif_export_all(ldif_dir):
    """exportation totale (premier lancement du script AAF)
    """
    sys.stderr.write("Lecture des établissements...\n")
    siren = {}
    uai = {}
    with open(join(ldif_dir, '01-etablissements.ldif'), 'w') as fh:
        chetab = CheetahTemplate.compile(file=join(template_path, "etablissement.ldif"))
        chservac = CheetahTemplate.compile(file=join(template_path, "servac.ldif"))
        for row in db.fetchtable('etablissement'):
            dico = translator.translate_one('etablissement', row)
            siren[dico['ENTStructureJointure_ori']] = dico['ENTStructureSIREN']
            uai[dico['ENTStructureJointure_ori']] = dico['ENTStructureUAI']
            dico = clean_empty_keys(dico)
            try:
                #Répartition des structures entre ENTEtablissement et ENTServAc
                #en fonction de l'attribut ENTStructureTypeStruct
                #cf. Annexe4 (Alimentation depuis le SI MEN) version 4.2
                if dico['ENTStructureTypeStruct'] in entservac_structures:
                    fh.write(str(chservac(searchList=[dico])))
                else:
                    fh.write(str(chetab(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='etablissement',
                            fromxml=False, db=db, id_column='id')
            except Exception as err:

                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (1): {0} pour l'etablissement {1}"
                msg = msg.format(str(err), row['id'])
                log_error('etablissement', 'CREATE_DONE', datestr, db, row['id'], msg)

        db.commit(None)

    sys.stderr.write("Lecture des classes...\n")
    with open(join(ldif_dir, '02-classes.ldif'), 'w') as fh:
        ch = CheetahTemplate.compile(file=join(template_path, 'classe.ldif'))
        for row in db.fetchtable('classe'):
            dico = translator.translate_one('classe', row)
            dico = clean_empty_keys(dico)
            try:
                dico['cn'] = gen_group_cn(dico['description_ori'], siren)
                dico['owner'] = gen_group_owner(dico['description_ori'], uai)
            except ValueError as err:
                # classe non référencée (#7199)
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (2): {0} pour la classe {1}"
                msg = msg.format(str(err), row['description'])
                log_error('classe', 'CREATE_DONE', datestr, db, row['description'],
                          msg)
                continue
            try:
                fh.write(str(ch(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='classe',
                            fromxml=False, db=db, id_column='description')
            except Exception as err:
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (3): {0} pour la classe {1}"
                msg = msg.format(str(err), row['description'])
                log_error('classe', 'CREATE_DONE', datestr, db, row['description'],
                          msg)

    db.commit(None)
    sys.stderr.write("Lecture des groupes...\n")
    with open(join(ldif_dir, '03-groupes.ldif'), 'w') as fh:
        ch = CheetahTemplate.compile(file=join(template_path, "groupe.ldif"))
        for row in db.fetchtable('groupe'):
            dico = translator.translate_one('groupe', row)
            dico = clean_empty_keys(dico)
            try:
                dico['cn'] = gen_group_cn(dico['description_ori'], siren)
                dico['owner'] = gen_group_owner(dico['description_ori'], uai)
            except ValueError as err:
                # établissement non référencé (#7199)
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (4): {0} pour le groupe {1}"
                msg = msg.format(str(err), row['description'])
                log_error('groupe', 'CREATE_DONE', datestr, db, row['description'],
                          msg)
                continue
            try:
                fh.write(str(ch(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='groupe',
                            fromxml=False, db=db, id_column='description')
            except Exception as err:
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (5): {0} pour le groupe {1}"
                msg = msg.format(str(err), row['description'])
                log_error('groupe', 'CREATE_DONE', datestr, db, row['description'],
                          msg)
    db.commit(None)
    sys.stderr.write("Lecture des élèves...\n")

    with open(join(ldif_dir, '04-eleves.ldif'), 'w') as fh:
        ch = CheetahTemplate.compile(file=join(template_path, "eleve.ldif"))
        # liens vers des responsables directement dans l'entrée élève
        # attributs optionnels selon le schéma ldap v1.52b
        fixmekeys = ['ENTEleveParents', 'ENTEleveAutoriteParentale',
                     'ENTElevePersRelEleve1', 'ENTEleveQualitePersRelEleve1',
                     'ENTElevePersRelEleve2', 'ENTEleveQualitePersRelEleve2']
        for row in db.fetchtable('eleve'):
            dico = translator.translate_one('eleve', row)
            # FIXME XXX HAAAACK pour autoriser le prénom vide alors qu'il
            # est supposé être obligatoire dans les préconisations du SDET.
            if dico['givenName'] is None:
                dico['givenName'] = "-"
            if dico['sn'] is None:
                dico['sn'] = "-"
            dico['cn'] = gen_user_cn(dico['sn'], dico['givenName'])
            dico['displayName'] = gen_user_displayname(dico['sn'], dico['givenName'])
            #dico['userPassword'] = pwgen()
            # FIXME: should it be provided ?
            if 'ENTEleveMajeur' not in dico:
                dico['ENTEleveMajeur'] = "N"
            # FIXME: optimization needed ?
            dico = clean_empty_keys(dico)

            for key in fixmekeys:
                if key in dico:
                    del(dico[key])
            try:
                fh.write(str(ch(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='eleve',
                            fromxml=False, db=db, id_column='id')
            except Exception as err:
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (6): {0} pour l'élève {1}"
                msg = msg.format(str(err), row['id'])
                log_error('eleve', 'CREATE_DONE', datestr, db, row['id'], msg)

    db.commit(None)

    sys.stderr.write("Lecture des responsables...\n")

    if import_responsable:
        with open(join(ldif_dir, '05-responsable.ldif'), 'w') as fh:
            ch = CheetahTemplate.compile(file=join(template_path, "responsable.ldif"))
            for row in db.fetchtable('responsable'):
                dico = translator.translate_one('responsable', row)
                try:
                    # FIXME XXX HAAAACK pour autoriser le prénom vide alors qu'il
                    # est supposé être obligatoire dans les préconisations du SDET.
                    if dico['givenName'] is None:
                        dico['givenName'] = "-"
                    if dico['sn'] is None:
                        dico['sn'] = "-"
                    dico['cn'] = gen_user_cn(dico['sn'], dico['givenName'])
                    dico['displayName'] = gen_user_displayname(dico['sn'], dico['givenName'])
                    #dico['userPassword'] = pwgen()
                    dico = clean_empty_keys(dico)
                    fh.write(str(ch(searchList=[dico])))
                    modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='responsable',
                                fromxml=False, db=db, id_column='id')
                except Exception as err:
                    datestr = make_datetime()
                    msg = "Erreur à la génération du LDIF (7): {0} pour le responsable {1}"
                    msg = msg.format(str(err), row['id'])
                    log_error('responsable', 'CREATE_DONE', datestr, db, row['id'],
                              msg)

    db.commit(None)

    sys.stderr.write("Lecture des enseignants...\n")
    with open(join(ldif_dir, '06-enseignants.ldif'), 'w') as fh:
        ch = CheetahTemplate.compile(file=join(template_path, "enseignant.ldif"))
        for row in db.fetchtable('enseignant'):
            dico = translator.translate_one('enseignant', row)
            # FIXME XXX HAAAACK pour autoriser le prénom vide alors qu'il
            # est supposé être obligatoire dans les préconisations du SDET.
            if dico['givenName'] is None:
                dico['givenName'] = "-"
            if dico['sn'] is None:
                dico['sn'] = "-"
            dico['cn'] = gen_user_cn(dico['sn'], dico['givenName'])
            dico['displayName'] = gen_user_displayname(dico['sn'], dico['givenName'])
            #dico['userPassword'] = pwgen()
            dico = clean_empty_keys(dico)
            try:
                fh.write(str(ch(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='enseignant',
                            fromxml=False, db=db, id_column='id')
            except Exception as err:
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (8): {0} pour l'enseignant {1}"
                msg = msg.format(str(err), row['id'])
                log_error('enseignant', 'CREATE_DONE', datestr, db, row['id'], msg)

    db.commit(None)

    sys.stderr.write("Lecture des personnels non enseignant...\n")
    with open(join(ldif_dir, '07-personnels.ldif'), 'w') as fh:

        # /!\ Il y'a deux types de personnel non enseignant /!\
        # * nonens-servac : personnes Non enseignants rattachés administrativement aux services académiques
        # * nonens-etab : personnes Non enseignants rattachés administrativement à un établissement d’enseignement
        chetab = CheetahTemplate.compile(file=join(template_path, "nonens-etab.ldif"))
        chservac = CheetahTemplate.compile(file=join(template_path, "nonens-servac.ldif"))
        for row in db.fetchtable('administratif'):
            dico = translator.translate_one('administratif', row)

            # FIXME XXX HAAAACK pour autoriser le prénom vide alors qu'il
            # est supposé être obligatoire dans les préconisations du SDET.
            if dico['givenName'] is None:
                dico['givenName'] = "-"
            if dico['sn'] is None:
                dico['sn'] = "-"
            dico['cn'] = gen_user_cn(dico['sn'], dico['givenName'])
            dico['displayName'] = gen_user_displayname(dico['sn'], dico['givenName'])
            #dico['userPassword'] = pwgen()
            dico = clean_empty_keys(dico)
            # le choix est déterminé par la présence d'une "structure de rattachement"
            # /!\ si dico['ENTPersonStructRattach'] vaut None, la clé est supprimée par "clean_empty_keys"
            try:
                if 'ENTPersonStructRattach' in dico:
                    fh.write(str(chetab(searchList=[dico])))
                else:
                    fh.write(str(chservac(searchList=[dico])))
                modif_fields(row, sqlgen, modif_type="CREATE_DONE", tablename='administratif',
                            fromxml=False, db=db, id_column='id')
            except Exception as err:
                import traceback
                traceback.print_exc()
                datestr = make_datetime()
                msg = "Erreur à la génération du LDIF (9): {0} pour l'administratif {1}"
                msg = msg.format(str(err), row['id'])
                log_error('administratif', 'CREATE_DONE', datestr, db, row['id'], msg)

    db.commit(None)


def ldif_export_maj(ldif_dir):
    """point d'entrée de l'insertion des deltas dans le ldap, à savoir:

    - les ajouts (DELTACREATE)
    - les modifications (DELTAUPDATE)
    - les suppression (DELTADELETE)

    Les ldif des élèves, profs, responsables, etc sont poussés dans trois
    fichiers en fonction des actions effectuées :

    - create -> pour ldapadd
    - modify -> pour ldapmodify
    - delete -> pour ldapdelete

    """
    ldif_export_eleves(ldif_dir)
    ldif_export_etablissements(ldif_dir)
    ldif_export_responsables(ldif_dir)
    ldif_export_enseignants(ldif_dir)
    ldif_export_administratifs(ldif_dir)


def ldif_export_eleves(ldif_dir):
    """
    templatisation des ldif élève
    """
    log.info("delta des élèves...")
    ldif_export_factory('DELTACREATE', 'eleve', ldif_dir, 'eleve.ldif',
                        ldap_delta_create_filename)
    ldif_export_factory('DELTAUPDATE', 'eleve', ldif_dir, 'maj_eleve.ldif',
                        ldap_delta_update_filename)
    ldif_delete('DELTADELETE', 'eleve', ldif_dir, ldap_delta_delete_filename)


def ldif_export_etablissements(ldif_dir):
    """
    templatisation des ldif établissement
    """
    log.info("deltas des établissements...")
    ldif_export_factory('DELTACREATE', 'etablissement', ldif_dir,
                        'etablissement.ldif', ldap_delta_create_filename)
    ldif_export_factory('DELTAUPDATE', 'etablissement', ldif_dir,
                        'maj_etablissement.ldif', ldap_delta_update_filename)
    # Remarque : Pas de suppression d'établissement prévue à ce jour


def ldif_export_responsables(ldif_dir):
    """
    templatisation des ldif responsable
    """
    log.info("deltas des responsables...")
    ldif_export_factory('DELTACREATE', 'responsable', ldif_dir,
                        'responsable.ldif', ldap_delta_create_filename)
    ldif_export_factory('DELTAUPDATE', 'responsable', ldif_dir,
                        'maj_responsable.ldif', ldap_delta_update_filename)
    ldif_delete('DELTADELETE', 'responsable', ldif_dir, ldap_delta_delete_filename)


def ldif_export_enseignants(ldif_dir):
    """
    templatisation des ldif enseignant
    """
    log.info("deltas des enseignants...")
    ldif_export_factory('DELTACREATE', 'enseignant', ldif_dir,
                        'enseignant.ldif', ldap_delta_create_filename)
    ldif_export_factory('DELTAUPDATE', 'enseignant', ldif_dir,
                        'maj_enseignant.ldif', ldap_delta_update_filename)
    ldif_delete('DELTADELETE', 'enseignant', ldif_dir, ldap_delta_delete_filename)


def ldif_export_administratifs(ldif_dir):
    """
    templatisation des ldif administratif
    """
    log.info("deltas des administratifs...")
    ldif_export_factory('DELTACREATE', 'administratif', ldif_dir,
                        'nonens-etab.ldif', ldap_delta_create_filename)
    ldif_export_factory('DELTAUPDATE', 'administratif', ldif_dir,
                        'maj_nonens-etab.ldif', ldap_delta_update_filename)
    ldif_delete('DELTADELETE', 'administratif', ldif_dir, ldap_delta_delete_filename)


# ____________________________________________________________________________________


def ldif_export_factory(export_type, dbname, ldif_dir, ldif_template, ldif_filename):
    """factory d'exportation en ldiff pour chaque table et chaque type

    :param export_type:  trois cas possibles,
                         `('DELTAUPDATE', 'DELTADELETE', 'DELTACREATE')`
    :param dbname: nom de la table ('eleve', 'responsable', ...)
    :param ldif_dir: dossier destination du ldif
    :param ldif_template: modele ldif utilisé, par exemple,
                          pour 'eleve' -> "maj_eleve.ldif"
    :param ldif_filename: see in config something like ldap_delta_update_filename
    """
    if export_type not in ('DELTAUPDATE', 'DELTADELETE', 'DELTACREATE'):
        msg = "le type d'exportation ldif demandé est inconnu : {0}".format(
              export_type)
        log.info(msg)
        return
        #raise TypeError(msg)

    ch = CheetahTemplate.compile(file=join(template_path, ldif_template))
    if dbname == "eleve":
        # liens vers des responsables directement dans l'entrée élève
        # attributs optionnels selon le schéma ldap v1.52b
        fixmekeys = ['ENTEleveParents', 'ENTEleveAutoriteParentale',
                 'ENTElevePersRelEleve1', 'ENTEleveQualitePersRelEleve1',
                 'ENTElevePersRelEleve2', 'ENTEleveQualitePersRelEleve2']
    else:
        fixmekeys = []
    # /!\ Il y a deux types de personnel non enseignant /!\
    # * nonens-servac : personnes Non enseignants rattachés administrativement aux services académiques
    # * nonens-etab : personnes Non enseignants rattachés administrativement à un établissement d’enseignement
    if dbname == "administratif":
        if export_type == 'DELTACREATE':
            chservac = CheetahTemplate.compile(file=join(template_path, 'nonens-servac.ldif'))
        else:
            chservac = CheetahTemplate.compile(file=join(template_path, 'maj_nonens-servac.ldif'))
    # /!\ Il y a deux types d'établissements /!\
    # * etablissement : établissement d'enseignement
    # * servac : service académique
    if dbname == "etablissement":
        if export_type == 'DELTACREATE':
            chservac = CheetahTemplate.compile(file=join(template_path, 'servac.ldif'))
        else:
            chservac = CheetahTemplate.compile(file=join(template_path, 'maj_servac.ldif'))

    with open(join(ldif_dir, ldif_filename), 'a+') as fh:
        sqlwhere = "WHERE FieldActionType = '{0}'".format(export_type)
        for row in db.fetchall(dbname, sqlwhere):
            dico = translator.translate_one(dbname, row)
            if export_type == 'DELTACREATE':
                #ENTPersonLogin: Login (identifiant de connexion à l'ENT)
                #ne pas modifier sur un update
                if dbname == "eleve" and 'ENTEleveMajeur' not in dico:
                    #ENTEleveMajeur: attribut obligatoire pas toujours fourni
                    dico['ENTEleveMajeur'] = "N"
            if dbname != "etablissement":
                # FIXME XXX HAAAACK pour autoriser le prénom vide alors qu'il
                # est supposé être obligatoire dans les préconisations du SDET.
                if dico['givenName'] is None:
                    dico['givenName'] = "-"
                if dico['sn'] is None:
                    dico['sn'] = "-"
                dico['cn'] = gen_user_cn(dico['sn'], dico['givenName'])
                dico['displayName'] = gen_user_displayname(dico['sn'], dico['givenName'])
            dico = clean_empty_keys(dico)
            for key in fixmekeys:
                if key in dico:
                    del(dico[key])
            try:
                if dbname == "administratif" and 'ENTPersonStructRattach' not in dico:
                    # le choix est déterminé par la présence d'une "structure de rattachement"
                    # /!\ si dico['ENTPersonStructRattach'] vaut None, la clé est supprimée par "clean_empty_keys"
                    fh.write(str(chservac(searchList=[dico])))
                elif dbname == "etablissement" and dico['ENTStructureTypeStruct'] in entservac_structures:
                    #Répartition des structures entre ENTEtablissement et ENTServAc
                    #en fonction de l'attribut ENTStructureTypeStruct
                    fh.write(str(chservac(searchList=[dico])))
                else:
                    fh.write(str(ch(searchList=[dico])))
                modif_export_type = export_type + "_DONE"
                modif_fields(row, sqlgen, modif_type=modif_export_type, tablename=dbname,
                            fromxml=False, db=db, id_column='id')
            except Exception as err:
                log.error("Erreur sur un {1} : {0}".format(dico, dbname))
                log.error(err)
        db.commit(None)


def ldif_delete(export_type, dbname, ldif_dir, ldif_filename):
    """Génération des instructions ldif de suppression de compte

    :param export_type: normalement un seul cas possible : 'DELTADELETE'
    :param dbname: nom de la table ('eleve', 'responsable', ...)
    :param ldif_dir: dossier destination du ldif
    :param ldif_filename: see in config something like ldap_delta_update_filename
    """
    if export_type not in ('DELTADELETE'):
        msg = "le type d'exportation ldif demandé est inconnu : {0}".format(
              export_type)
        log.info(msg)
        return
    model = """dn: uid=%s,{0}
changetype: delete

""".format(usersdn)
    with open(join(ldif_dir, ldif_filename), 'a+') as fh:
        sqlwhere = "WHERE FieldActionType = '{0}'".format(export_type)
        for row in db.fetchall(dbname, sqlwhere):
            dico = translator.translate_one(dbname, row)
            try:
                fh.write(model % dico['uid'])
                modif_export_type = export_type + "_DONE"
                modif_fields(row, sqlgen, modif_type=modif_export_type, tablename=dbname,
                            fromxml=False, db=db, id_column='id')
            except Exception as err:
                log.error("Erreur à la suppression d'un {1} : {0}".format(dico, dbname))
                log.error(err)
        db.commit(None)


# ____________________________________________________________________________________

def regen_slapd_config():
    engine = CreoleTemplateEngine()

    slapd = {'source': '/usr/share/eole/creole/distrib/slapd.conf',
             'name': '/etc/ldap/slapd.conf',
             'full_name': '/etc/ldap/slapd.conf',
             'activate' : True,
             'del_comment': '',
             'mkdir' : False,
             'rm' : False,
    }
    db_config = {'source': '/usr/share/eole/creole/distrib/DB_CONFIG',
                 'name': '/var/lib/ldap/DB_CONFIG',
                 'full_name': '/var/lib/ldap/DB_CONFIG',
                 'activate' : True,
                 'del_comment': '',
                 'mkdir' : False,
                 'rm' : False,
    }
    str(slapd['source'])
    group = client.get_container_infos('root')
    engine.creole_variables_dict['import_slapadd'] = True
    engine.creole_variables_dict['ldap_cachesize'] = 100000
    engine.process(slapd, group)
    engine.process(db_config, group)


if __name__ == "__main__":
    if sys.argv[1] == 'all':
        ldif_export_all(sys.argv[2])
        regen_slapd_config()
    elif sys.argv[1] == 'delta':
        ldif_export_maj(sys.argv[2])
    else:
        sys.stderr.write("bad argument")
        sys.exit(1)
