# -*- coding: UTF-8 -*-
import os
import datetime
import time
from pwd import getpwnam

# xml_rpc
from twisted.web import xmlrpc

from twisted.python import log
log.FileLogObserver.timeFormat = "%Y/%m/%d %H:%M:%S %Z"

from scribe.eoleldap import Ldap
from scribe.linker import _user_factory
from scribe import eoleuser
from scribe import eolegroup
from scribe.eleves import Eleve
from scribe.enseignants import Enseignant
from scribe.errors import LdapExistingUser
from scribe.ldapconf import SUPPORT_ETAB, PROF_FILTER
from pyeole import process

from fichier.passwd import check_nb_min_classes

import devoirs
import connexions
import ldap_utils

from config import debug


class RpcServer(xmlrpc.XMLRPC):
    def __init__(self, secret_key):
        """Prend la clé secrète en paramètre
        """
        xmlrpc.XMLRPC.__init__(self)
        self.secret_key = secret_key

    def verif_cle(self, cle):
        """Vérifie une clé en la comparant à celle dans config
        """
        return str(cle) == self.secret_key

    def est_prof(self, username):
        """Vérifie si username est un prof
        """
        return ldap_utils.is_prof(username)

    def get_prof_class(self, login):
        """ renvoie les classes du prof login """
        classes = eolegroup.Group().get_classes()
        return self.filter_class_by_prof(login, classes)

    def get_prof_class_list(self, login):
        """ renvoie la liste des classes d'un prof """
        cls = []
        for classe in self.get_prof_class(login).values():
            cls.extend(classe)
        return cls

    def filter_class_by_prof(self, prof, classes):
        """ renvoie les classes du prof prof """
        groups = eoleuser.User().get_attr(prof, 'Divcod')
        ret_classes = classes.copy()
        for niveau, classe_list in classes.items():
            nivo = []
            for classe in classe_list:
                if classe in groups:
                    nivo.append(classe)
            if nivo != []:
                ret_classes[niveau] = nivo
            else:
                ret_classes.pop(niveau)
        return ret_classes

    def xmlrpc_remote_is_eleve_from_prof(self, cle, eleve, prof):
        """is eleve a member of a class administrated by prof"""
        ldapuser = eoleuser.User()
        ldapuser.ldap_admin.connect()
        if ldapuser._get_attr(eleve, 'typeadmin'):
            #c'est un prof
            ldapuser.ldap_admin.close()
            return False
        cls = self.get_prof_class_list(prof)
        for classe in cls:
            if eleve in ldapuser._get_members(classe):
                ldapuser.ldap_admin.close()
                return True
        ldapuser.ldap_admin.close()
        return False

    def xmlrpc_remote_est_prof(self, cle, username):
        """renvoie True si le username est prof et False sinon
        """
        if debug:
            log.msg('Appel de la fonction remote_est_prof pour %s' % username)
        return self.est_prof(username)

    def xmlrpc_remote_est_admin(self, username):
        """renvoie True si le username est admin et False sinon
        """
        if debug:
            log.msg('Appel de la fonction est_admin pour %s' % username)
        groups = ldap_utils.get_user_groups(username)
        if 'DomainAdmins' in groups:  # Ne prend pas en compte le groupe 'Domain Admins' (seth, amonecole)
            return True
        return False

    def xmlrpc_remote_est_admin_etab(self, username):
        """renvoie True si le username est etab_admin d'au moins 1 etab et False sinon
        """
        if debug:
            log.msg('Appel de la fonction est_admin_etab pour %s'%username)
        etabs = ldap_utils._get_etabs_from_user(username)
        if not etabs:
            # Mode mono-etab, check si le user est admin
            return self.xmlrpc_remote_est_admin(username)
        groups = ldap_utils.get_user_groups(username)
        groups = [group.lower() for group in groups]
        for etab in etabs:
            if "admin-" + etab.lower() in groups:
                return True
        return False

    def xmlrpc_remote_groupes(self, typegroup, username=None):
        """renvoie la liste des groupes
        """
        if debug:
            log.msg('Appel de la fonction remote_groupes pour %s'%typegroup)
        return ldap_utils.get_groupes(typegroup, user=username)

    def xmlrpc_remote_user_groupes(self, user):
        """renvoie la liste des groupes
        """
        if debug:
            log.msg('Appel de la fonction get_user_groups pour %s'%user)
        return ldap_utils.get_user_groups(user)

    def xmlrpc_remote_membres(self, groupe, rgroupe=None):
        """renvoie la liste des membres de 'groupe'
        """
        if debug:
            log.msg('Appel de la fonction remote_membres pour %s (filtre : %s)'%(groupe, rgroupe))
        if rgroupe: return ldap_utils.get_membres_croise([groupe, rgroupe])
        else: return ldap_utils.get_membres(groupe)

    def xmlrpc_remote_etabs_eleves(self, etabs):
        """renvoie la liste des membres d'un ou plusieurs établissements
        """
        if (isinstance(etabs, list)): #si multietablissement
            if debug:
                log.msg('Appel de la fonction xmlrpc_remote_etabs_eleves pour etablissements %s ' % (etabs))
            else:
                return ldap_utils.get_membres_croise_etab(etabs, 'eleves')
        else:
            a=Eleve()
            if debug:
                log.msg('Appel de la fonction xmlrpc_remote_etabs_eleves pour etablissement %s '% (etabs))
            else:
                return a.get_members('eleves')

    def xmlrpc_remote_etabs_profs(self, etabs):
        """renvoie la liste des membres d'un ou plusieurs établissements
        """
        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_etabs_profs pour etablissement %s '%(etabs))
            return

        profs = []
        if SUPPORT_ETAB:
            if not isinstance(etabs, list):
                etabs = [etabs]
            profs = ldap_utils.get_membres_croise_etab(etabs, 'professeurs')
        else:
            a = Enseignant()
            profs = a.get_members('professeurs')

        if 'admin' in profs:
            profs.remove('admin')

        return profs

    def xmlrpc_remote_etabs_group_members(self, etabs, group):
        """renvoie la liste des membres d'un ou plusieurs établissements d'un group
        """
        if (isinstance(etabs, list)): #si multietablissement
            if debug:
                log.msg('Appel de la fonction xmlrpc_remote_etabs_group_members pour etablissement %s '%(etabs))
            else:
                return ldap_utils.get_membres_croise_etab(etabs,group)
        else:
            a=Enseignant()
            if debug:
                log.msg('Appel de la fonction xmlrpc_remote_etabs_group_members pour etablissement %s '%(etabs))
            else:
                return a.get_members(group)

    def xmlrpc_add_etab_prof(self, login, pwd, nom, prenom, group, profil, etab, temporary=False, tempcount=0):
        prof = {}
        ldapprof = Enseignant()
        ldapprof.ldap_admin.connect()

        if not ldapprof._is_group(group):
            if group.startswith('invite-'):
                return "Le groupe %s n'existe pas, veuillez le créer dans l'EAD2" % (group)
            return "Le groupe %s n'existe pas" % (group)
        prof['login'] = login
        prof['password'] = pwd
        prof['prenom'] = prenom
        prof['nom'] = nom
        prof['groups'] = group
        prof['etab'] = etab
        # prof['profil'] = profil
        if temporary:
            if tempcount > 0:
                tempcount += 1  # We add a day to fully cover the last day
            tempcount += 1  # Hack: add 1 because EAD add 1 day too
            prof['temporary'] = tempcount
        # Paramètres forcés
        prof['date'] = '01/01/0000'
        prof['civilite'] = '1' # Mr
        try:
            ldapprof._add(**prof)
        except LdapExistingUser:
            ldapprof.ldap_admin.close()
            return "Erreur : Le login %(login)s n'est pas disponible" % prof
        except Exception as msg:
            ldapprof.ldap_admin.close()
            return "Erreur : %s" % str(msg)

        ldapprof.ldap_admin.close()

        return ""

    def xmlrpc_expire_prof(self, login, expire):
        prof = {}
        ldapprof = Enseignant()
        ldapprof.ldap_admin.connect()

        if login == '(vide)' or not ldapprof.is_enseignant(login):
            return "L'utilisateur %s n'existe pas"%(login)
        ldapprof.ldap_admin.close()

        expire = int(expire)
        if expire > 0:
            expire += 1  # We add a day to fully cover the last day
        cmd = "/usr/sbin/smbldap-usermod --shadowExpire {} {}".format(expire, login)
        ret_code, value, ret_value = process.system_out(['at', 'now'], stdin=cmd)

        return True


    def xmlrpc_remote_get_devoirs(self, cle, username):
        """renvoie la liste des devoirs de 'username'
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_get_devoirs pour %s'%(username))
        return devoirs.get_devs(username)

    def xmlrpc_remote_distribute(self, cle, username, groupe, dev_name, eleve_only, in_perso):
        """distribue "dev_name" du prof "username" à "groupe"
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_distribute pour : username=%s, groupe=%s, dev_name=%s, eleve_only=%s, in_perso=%s'%(username, groupe, dev_name, eleve_only, in_perso))
        return devoirs.distribute(username, groupe, dev_name, eleve_only, in_perso)

    def xmlrpc_remote_ramasse(self, cle, username, dev_name):
        """rammasse le devoir "dev_name" du prof "username"
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_ramasse pour : username=%s, dev_name=%s'%(username, dev_name))
        return devoirs.ramasse(username, dev_name)

    def xmlrpc_remote_rendre(self, cle, username, dev_name):
        """rend la correction du devoir "dev_name" du prof "username"
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_rendre pour : username=%s, dev_name=%s'%(username, dev_name))
        return devoirs.rendre(username, dev_name)

    def xmlrpc_remote_supprimer(self, cle, username, dev_name):
        """Supprime les données du devoir "dev_name" du prof "username"
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_supprimer pour : username=%s, dev_name=%s'%(username, dev_name))
        return devoirs.supprimer(username, dev_name)

    def xmlrpc_remote_copie_fichier(self, cle, username, dev_name, emplacement, nom_fichier, donnees_fichier):
        """Réception et écriture d'un fichier vers /home/workgroups/devoirs/<username>/<dev_name> selon son <emplacement> ("devoir" ou "data", "devoir" = considéré comme valeur par défaut)
           Si <emplacement> = "devoir" alors écriture dans /home/workgroups/devoirs/<username>/<dev_name>
           Si <emplacement> = "data" alors écriture dans /home/workgroups/devoirs/<username>/<dev_name>/donnees
           <donnees_fichier> = objet xmlrpclib.Binary
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction remote_copie_fichiers_devoir pour : username=%s, dev_name=%s, emplacement=%s, nom_fichier=%s'%(username, dev_name, emplacement, nom_fichier))

        # emplacement = "devoir" est considéré comme valeur par défaut
        dirprofdev = os.path.join("/home/workgroups/devoirs", username, dev_name)
        if not os.path.isdir(dirprofdev): os.makedirs(dirprofdev)

        if emplacement == "data":
            dirprofdev = os.path.join(dirprofdev, "donnees")
            if not os.path.isdir(dirprofdev): os.makedirs(dirprofdev)

        chemin_fichier = os.path.join(dirprofdev, nom_fichier)

        with open(chemin_fichier, "wb") as handle:
            handle.write(donnees_fichier.data)

        os.system('setfacl -PRm u:%s:rwx %s'%(username, dirprofdev))

        return True

    def xmlrpc_remote_get_dev_todel(self, cle, username):
        """Retourne la liste des devoirs pour un prof dont les données sont tjrs sur le serveur
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction remote_get_dev_todel pour %s'%(username))

        dirprofdev = os.path.join("/home/workgroups/devoirs", username)
        l = []
        if not os.path.exists(dirprofdev): return l
        for dev_name in os.listdir(dirprofdev):
            if dev_name.lower() == 'tmp': continue
            if os.path.isdir(os.path.join(dirprofdev, dev_name)):
                l.append('%s'%(dev_name))
        l.sort()
        return l

    def xmlrpc_remote_get_mail(self, cle, username):
        """Renvoi le mail de cet utilisateur
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_get_mail pour %s'%(username))

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(username, conn.connexion)
        mail = ldapuser.get_attr(username, "mail")
        if mail == []:
            return ""
        else:
            return mail[0]

    def xmlrpc_remote_get_nom(self, cle, username):
        """Renvoi le titre et le nom de cet utilisateur (ex: M. Trucmuche)
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_get_nom pour %s'%(username))

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(username, conn.connexion)
        return "%s %s" % (ldapuser.get_attr(username, "personalTitle")[0], ldapuser.get_attr(username, "sn")[0])

    def xmlrpc_remote_get_attr(self, cle, username, usertocheck, attribut):
        """Renvoi un attribut ldap pour l'utilisateur usertocheck
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return []

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_get_attr(%s, %s, %s)'%(username, usertocheck, attribut))

        conn = Ldap()
        conn.connect()
        try:
            ldapuser = _user_factory(username, conn.connexion)
            if isinstance(usertocheck, list):
                if attribut == "all":
                    attrs = ldapuser.users_get_attrs(usertocheck, ["sn", "givenName"])
                else:
                    attrs = ldapuser.users_get_attr(usertocheck, attribut)
            else:
                if attribut == "all":
                    attrs = ldapuser.get_attrs(usertocheck, ["sn", "givenName"])
                else:
                    attrs = ldapuser.get_attr(usertocheck, attribut)
        except Exception as e:
            log.msg(e)
            return []
        return attrs

    def xmlrpc_remote_check_nb_min_classes(self, cle, password, nb_classe_min):
        """Teste si le password contient bien nb_classe_min de classe de caractères
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return []

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_check_nb_min_classes')

        return check_nb_min_classes(password, nb_classe_min)

    def xmlrpc_remote_check_password(self, cle, username, password):
        """Teste si le password correspond bien
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_check_password pour %s' % username)

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(username, conn.connexion)

        return ldapuser.auth(username, password)

    def xmlrpc_remote_change_mdp(self, cle, username, old_password, new_password):
        """Change le mot de passe de username en vérifiant son ancien mdp
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_change_mdp')

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(username, conn.connexion)

        # si un prof change son mdp, il doit connaitre le précédent
        if not ldapuser.auth(username, old_password):
            return False

        # application du changement de mot de passe
        ldapuser.c_mod_password(username, new_password, container=None)
        conn.close()
        return True

    def xmlrpc_remote_change_mdp_eleve(self, cle, username, logineleve, new_password, must_change=False):
        """Changement du mot de passe d'un élève par un prof
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_change_mdp')

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(logineleve, conn.connexion)

        # application du changement de mot de passe
        ldapuser.c_mod_password(logineleve, new_password, container=None,  must_change=must_change)
        conn.close()
        return True

    def xmlrpc_remote_password_must_change(self, cle, username, logineleve):
        """Force le changement du mot de passe à la première connexion d'un élève
        """
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction xmlrpc_password_must_change par %s pour %s' % (username, logineleve))

        conn = Ldap()
        conn.connect()
        ldapuser = _user_factory(logineleve, conn.connexion)

        # application du changement de mot de passe
        ldapuser.password_must_change(logineleve, container=None)
        conn.close()
        return True

    def xmlrpc_remote_save_password_csv(self, cle, username, listemdp):
        """Enregistre les mots de passe élèves modifiés par le prof dans un fichier csv dans son répertoire perso
           listemdp = {"login1": "mdp1", "login2": "mdp2",...}
        """
        try:
            uid = getpwnam(username).pw_uid
        except BaseException:
            uid = 0
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        if debug:
            log.msg('Appel de la fonction xmlrpc_remote_save_password_csv pour %s' % username)

        user_rep = os.path.join('/home', username[0], username, 'perso')
        if not os.path.isdir(user_rep):
            return ""

        rootname = "mot-de-passe_%d_%d_" % (datetime.date.today().day, datetime.date.today().month)
        for increment in range(15):
            name = rootname + str(increment) + '.csv'
            filename = os.path.join(user_rep, name)
            if not os.path.isfile(filename):
                break
        if os.path.isfile(filename):
            return ""

        conn = Ldap()
        conn.connect()
        with open(filename, 'w') as pwd_fh:
            pwd_fh.write("CLASSE;NOM;PRENOM;LOGIN;MOT DE PASSE;\r\n")

            for login, mdp in listemdp.items():
                ldapuser = _user_factory(login, conn.connexion)
                prenom = ldapuser._get_attr(login, 'givenName')[0]
                nom = ldapuser._get_attr(login, 'sn')[0]
                classe = ldapuser._get_attr(login, 'Divcod')[0]
                pwd_fh.write("%s;%s;%s;%s;%s;\r\n" % (classe, nom, prenom, login, mdp))
        os.chown(filename, uid, -1)
        conn.close()
        return os.path.split(filename)[1]

    def xmlrpc_remote_get_connectes_groupes(self, cle, username, group):
        """Get connected eleves from group, return a list of login, hostname and IP
        """
        if debug:
            log.msg('Appel de la fonction remote_get_connectes_groupes par %s pour %s'%(username, group))
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False

        connexion = connexions.Connexions()
        connected = []
        for login, _, machine, _, ip in connexion.get_connected_by_group(group):
            connected.append({'login': login, 'machine': machine, 'ip': ip})
        return connected

    def xmlrpc_remote_get_list_profs(self, cle, username, classe):
        """Returns the prof list that are admin of <classe> from the ldap with some other informations
           When classe is None, all the prof and the classes admin by current user are listed
           => [{"login": "nom prenom",...}, {"class1": ["login1",...], ...}]
           When a classe is provided, only the list of the admin of this class are listed
           => {"class1": ["login1",...]}
           When classe is not in the list of classes administred by current user
           => False
        """
        if debug:
            log.msg('Appel de la fonction remote_get_list_profs par %s'%(username))
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return False
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return False
        if classe == "" :
            classe = None   # can't pass None through xmlrpc

        ens = Enseignant()  # direct call to Enseignant without using user_factory to be able to modify the PROF_FILTER
        ens.ldap_admin.connect()
        if classe is None:
            filtre = "(&{0})".format(PROF_FILTER)
        else:
            filtre = "(&{0}(Divcod={1}))".format(PROF_FILTER, classe)
        ret = ens._get_users(filtre, ['uid', 'sn', 'givenName', 'Divcod'])
        list_prof = {}
        # get list of classes administered by the current user
        #//list_user_classes = [ userclasse for prof in ret if prof["uid"][0]==username for userclasse in prof.get("Divcod", []) ]
        if username != "admin":
            list_user_classes = self.get_prof_class_list(username)
        else:
            # l'admin gère d'office toutes les classes
            list_user_classes = self.xmlrpc_remote_groupes("classe")
        list_classe = {user_class:[] for user_class in list_user_classes}
        for prof in ret:
            login = prof["uid"][0]
            if login != "admin" and login != username: # neither the admin nor the current user
                list_prof[login] = "{0} {1}".format(prof["sn"][0], prof["givenName"][0])
                if prof.get("Divcod"):
                    for divcod in prof["Divcod"]:
                        if classe is not None and divcod != classe:
                            continue
                        if divcod in list_user_classes: #only deals with classes administred by current user
                            list_classe[divcod].append(login)
        if classe:
            # filter list_classe with the only requested class
            if classe in list_classe:
                return {classe:list_classe[classe]}
            else:
                log.msg('Classe %s non trouvée dans les classes administrées par %s'%(classe, username))
                return False
        else:
            return [list_prof, list_classe]

    def xmlrpc_remote_set_admin_right_to_class(self, cle, username, login, classe):
        """Add admin right to a class for a login
        Returns True if all ok else an error message
        """
        if debug:
            log.msg('Appel de la fonction remote_set_admin_right_to_class par %s pour la classe %s et login %s'%(username, classe, login))
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return "Erreur Serveur"
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return "Erreur Serveur"
        elif login == "admin":
            log.msg("Impossible de changer les droits de l'admin")
            return "Erreur Serveur"
        elif login == username:
            log.msg("Impossible de se changer les droits à soi-même")
            return "Erreur Serveur"

        try:
            list_classes = self.get_prof_class_list(login)
            old_set = set(list_classes)
            list_classes.append(classe)
            new_set = set(list_classes)

            if new_set != old_set:
                ldapuser = _user_factory(login)
                ldapuser.set_attr(login, 'Divcod', list_classes)
                if list_classes != []:
                    ldapuser.set_attr(login, 'typeadmin', '2')
                else:
                    ldapuser.set_attr(login, 'typeadmin', '0')
        except Exception as e:
            log.msg(e)
            return str(e)
        return True

    def xmlrpc_remote_del_admin_right_for_class(self, cle, username, login, classe):
        """Delete admin right from a class for a login
        Returns True if all ok else an error message
        """
        if debug:
            log.msg('Appel de la fonction remote_set_admin_right_to_class par %s pour la classe %s et login %s'%(username, classe, login))
        if not self.verif_cle(cle) :
            log.msg('Clé erronée : %s'%cle)
            return "Erreur Serveur"
        elif not self.est_prof(username) :
            log.msg('User non prof : %s'%username)
            return "Erreur Serveur"
        elif login == "admin":
            log.msg("Impossible de changer les droits de l'admin")
            return "Erreur Serveur"
        elif login == username:
            log.msg("Impossible de se changer les droits à soi-même")
            return "Erreur Serveur"
        try:
            list_classes = self.get_prof_class_list(login)
            old_set = set(list_classes)
            new_set = set(list_classes)
            new_set.discard(classe)

            if new_set != old_set:
                ldapuser = _user_factory(login)
                ldapuser.set_attr(login, 'Divcod', list(new_set))
                if list(new_set) != []:
                    ldapuser.set_attr(login, 'typeadmin', '2')
                else:
                    ldapuser.set_attr(login, 'typeadmin', '0')
        except Exception as e:
            log.msg(e)
            return e
        return True
