# -*- coding: UTF-8 -*-

###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# ldap_utils.py
#
# Librairie pour la gestion des requetes LDAP
#
###########################################################################


import sys, ldap
from os.path import isdir
from os import makedirs
sys.path.append('/usr/share/eole/controlevnc')
import config

class Ldap:
    # FIXME 2.3
    def __init__(self, serveur=config.ldap_serveur):
        self.query = ldap.open(serveur)
        self.query.simple_bind("config.ldap_admin")
        #Creation du repertoire /usr/share/eole/config
        if not isdir(config.vareole):
            makedirs(config.vareole)

    def unbind(self):
        self.query.unbind()

    def get_groupe_liste(self, gtype=None):
        """renvoie la liste des groupes
        """
        sfilter = '(objectclass=sambaGroupMapping)(objectclass=posixGroup)'
        if gtype == 'groupe': sfilter = '(&%s(description=Groupe *))'%sfilter
        elif gtype == 'classe': sfilter = '(&%s(description=Classe *))'%sfilter
        elif gtype == 'equipe': sfilter = '(&%s(description=Equipe *))'%sfilter
        elif gtype == 'matiere': sfilter = '(&%s(description=Matiere *))'%sfilter
        elif gtype == 'niveau': sfilter = '(&%s(description=Niveau *))'%sfilter
        else: sfilter = '(&%s)'%sfilter
        result = self.query.search_s("config.ldap_base_dn", ldap.SCOPE_SUBTREE, sfilter)
        r = [res[1]['cn'][0] for res in result]
        r.sort()
        return r

    def get_membres(self, groupe):
        result_group = self.query.search_s("config.ldap_base_dn", 2, "(&(cn=%s))"%groupe)
        if len(result_group) == 0: raise Exception('Groupe %s non trouve'%groupe)
        # on créé la liste des noms de groupes
        try: liste_membres = result_group[0][1]['memberUid']
        except: liste_membres = []
        liste_membres.sort()
        return liste_membres

    def get_user_groups(self, user, assoc=True, banned=True):
        """récupère la liste des groupes dont 'user' fait partie si assoc=True
        sinon renvoie la liste des autres groupes
        la liste est filtrée sur bannedgrps si banned
        """
        groupes = self.get_groupe_liste()
        bannedgrps = ['DomainComputers']
        liste_group = []
        for group in groupes:
            if banned and group in bannedgrps: continue
            membres = self.get_membres(group)
            if assoc:
                if user in membres: liste_group.append(group)
            else:
                if user not in membres: liste_group.append(group)
        liste_group.sort()
        return liste_group

    def get_user_shares(self, user):
        """ fonction de récupération des partages associés à un utilisateur
        """
        groups = self.get_user_groups(user)
        shares = self.get_shares()
        share_user=[]
        for share in shares:
            sharepath, sharegroup, sharename = share
            if sharegroup in groups:
                share_user.append((sharename,sharepath))
        return share_user

    def get_user_homedrive(self, user):
        """récupération du chemin du répertoire perso d'un utilisateur
        """
        result_user = self.query.search_s("config.ldap_base_dn", 2,\
                "(&(objectclass=sambaSAMAccount)(objectclass=inetOrgPerson)(uid=%s))"%user)
        # création de la liste des noms récupérés
        # FIXME 2.3
        #dicoEole = parsedico.parse_dico()
        return (str(result_user[0][1]['sambaHomeDrive'][0]),"\\\\%s\\PERSO"%config.smb_netbios_name)

    def get_user_sid(self, user):
        """récupération du sid de l'utilisateur
        """
        try: sid = self.query.search_s("config.ldap_base_dn", 2, "(&(objectclass=sambaSAMAccount)(uid=%s))"%user)[0][1]['sambaSID'][0]
        except: raise Exception('Utilisateur %s non trouve'%user)
        return sid

    def get_user_long_name(self, user):
        """récupération du displayName de l'utilisateur
        """
        try: lname = self.query.search_s("config.ldap_base_dn", 2, "(&(objectclass=sambaSAMAccount)(uid=%s))"%user)[0][1]['displayName'][0]
        except: raise Exception('Utilisateur %s non trouve'%user)
        return lname

    def get_shares(self):
        """liste de tous les répertoires partagés
        """
        result_share = self.query.search_s("config.ldap_base_dn", 2, "(objectclass=sambaFileShare)")
        list_share = []
        for share in result_share:
            list_share.append((share[1]['sambaShareURI'][0], share[1]['sambaShareGroup'][0], share[1]['sambaShareName'][0],))
        return list_share

    def get_machines(self):
        """Renvoie la liste des machines inscrites dans l'annuaire (NT, 2kX, XP, Vista)
        """
        result_mach = self.query.search_s("config.ldap_base_dn", 2, "(description=Computer)")
        list_mach = []
        for machine in result_mach:
            n = machine[1]['uid'][0]
            if n.endswith('$'): n = n[:-1]
            list_mach.append(n)
        return list_mach


###########
# Groupes #
###########
def get_groupes(gtype=None, banned=True):
    banned_groups = ['DomainAdmins', 'DomainComputers', 'DomainUsers', 'PrinterAdmins', 'PrintOperators']
    l = Ldap()
    result_dispo = l.get_groupe_liste(gtype)
    l.unbind()
    liste_group = []
    for groupe in result_dispo:
        if banned and groupe in banned_groups: continue
        liste_group.append(groupe)
    liste_group.sort()
    return liste_group

def get_classes():
    return get_groupes('classe')

def get_niveau():
    return get_groupes('niveau')

def get_classes_et_groupes():
    r = get_classes()
    r.extend(get_groupes('groupe'))
    return r

def get_membres(groupe):
    l = Ldap()
    r = l.get_membres(groupe)
    l.unbind()
    return r

def get_nb_membres(groupe):
    if type(groupe) == list or  type(groupe) == tuple:
        return len(get_membres_croise(groupe))
    else:
        return len(get_membres(groupe))

def get_membres_croise(groupes):
    """renvoie les membres de groupe qui sont aussi dans rgroupe
    """
    g = get_membres(groupes[0])
    for i in groupes[1:]:
        m = get_membres(i)
        g = list(set(g).intersection(set(m)))
    return g

################
# utilisateurs #
################
def get_user_groups(user):
    l = Ldap()
    r = l.get_user_groups(user)
    l.unbind()
    return r

def get_user_shares(user):
    l = Ldap()
    r = l.get_user_shares(user)
    l.unbind()
    return r

def get_user_homedrive(user):
    l = Ldap()
    r = l.get_user_homedrive(user)
    l.unbind()
    return r

def get_user_sid(user):
    l = Ldap()
    r = l.get_user_sid(user)
    l.unbind()
    return r

def get_user_long_name(user):
    l = Ldap()
    r = l.get_user_long_name(user)
    l.unbind()
    return r

def is_prof(user):
    if 'professeurs' in get_user_groups(user): return True
    else :return False

def is_eleve(user):
    if 'eleves' in get_user_groups(user): return True
    else :return False

############
# partages #
############
def get_shares():
    l = Ldap()
    r = l.get_shares()
    l.unbind()
    return r

############
# Machines #
############
def get_machines():
    l = Ldap()
    r = l.get_machines()
    l.unbind()
    return r
