# -*- coding: UTF-8 -*-

###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# devoirs.py
#
# Librairie de gestion de la distribution de devoirs
#
# distribution par un utilisateur aux membres d'un groupe avec choix :
#   * qu'aux élèves du groupe
#   * soit dans le dossier $HOME/perso/devoirs/<login_prof>/<nom_devoir>/
#   * soit dans le dossier $HOME/devoirs/<login_prof>/<nom_devoir>/
#  impératifs :
#   * les fichiers du devoirs et les "donnees" doivent se trouver dans
#      /home/workgroups/devoirs/<login_prof>/<nom_devoir>/
#   ce dossier est accessible par les liens
#      $HOME/perso/devoirs/.distribues
#      $HOME/devoirs/.distribues
#
# ramasse <nom_du_devoir> par un utilisateur dans
#  <login_prof>/perso/devoirs/rammasses/<nom_devoir>/<login_eleve>/
#
# rend <nom_du_devoir> par un utilisateur dans
#  <login__eleve>/perso/devoirs/<login_prof>/<nom_devoir>/correction
#
# supprime les données :
#  * suppression du répertoire /home/workgroups/devoirs/<login_prof>/<nom_devoir>/
#
# les informations sont stockées dans
#  /home/workgroups/devoirs/<login_prof>/devoirs.dat
#
###########################################################################


import sys, os, shutil, pickle, time, string
sys.path.append('/usr/share/eole/controlevnc')
import ldap_utils
from scribe.ldapconf import SUPPORT_ETAB, REVERSE_LOGIN_DEVOIR
from scribe.eoletools import get_reverse_login

from twisted.python import log
# FIXME 2.3
#from creole import parsedico
#DICO_EOLE = parsedico.parse_dico()
#HOME_PATH = DICO_EOLE['home_path']
HOME_PATH = '/home'
# chemin vers le répertoire "perso" de l'utilisateur
PERSO_DIR = os.path.join(HOME_PATH, '%(user).1s', '%(user)s', 'perso')
# chemin vers le répertoire "devoirs" de l'utilisateur
DEV_BASE_DIR = os.path.join(HOME_PATH, '%(user).1s', '%(user)s', 'devoirs')
DEV_PERSO_BASE_DIR = os.path.join(HOME_PATH, '%(user).1s', '%(user)s', 'perso/devoirs')
# le dossier contenant les devoirs distribués par un prof.
DEV_DIST_ROOT_DIR = os.path.join(HOME_PATH, 'workgroups/devoirs/')
DEV_DIST_DIR = os.path.join(DEV_DIST_ROOT_DIR, '%s')
# les informations concernant les devoirs (nom, groupe, etc) sont sauvegardés
# dans le fichier /home/<l>/<login>/devoirs/.devoirs.dat
DEV_INFO = os.path.join(DEV_DIST_DIR, 'devoirs.dat')
RESERVED_NAMES = ['.distribues', '.ftp', 'Maildir', 'corrections', 'devoirs', 'devoir', 'donnees', 'groupes', 'perso', 'ramasses']
# séparateur pour le préfixage des noms de fichiers par les logins
SEP_PREFIX = '--'
FICHIER_NON_MODIFIE = 'NON_FAIT_'

def load_dict(fich):
    """charge devoirs.dat
    """
    try: return pickle.load(open(fich, 'rb'))
    except: return {}

def save_dict(fich, dico):
    """sauve devoirs.dat
    """
    return pickle.dump(dico, open(fich,'wb'))

def get_devs(username):
    fich = DEV_INFO%username
    return load_dict(fich)

def save_devs(username, devs):
    fich = DEV_INFO%username
    return save_dict(fich, devs)

def copy_dev(rep_path, user_dest_dir, user):
    """copie <rep_path>/* dans /home/<u>/<user>/perso/devoirs/<dev_name>
    crée un lien /home/<u>/<user>/perso/devoirs/<dev_name>/donnees => <rep_path>/donnees
    """
    if os.path.exists(user_dest_dir): shutil.rmtree(user_dest_dir)
    os.makedirs(user_dest_dir)
    for i in os.listdir(rep_path):
        f = os.path.join(rep_path, i)
        if os.path.isfile(f): shutil.copy(f, user_dest_dir)
        # copie les dossiers sauf le dossier "donnees"
        if os.path.isdir(f) and i.strip().lower() != 'donnees':
            shutil.copytree(f, user_dest_dir)
    # crée un lien vers /home/workgroup/<prof>/<devoir>/donnees
    if os.path.isdir(os.path.join(rep_path, 'donnees')):
        os.symlink(os.path.join(rep_path, 'donnees'), os.path.join(user_dest_dir, 'donnees'))


def suppr_devoirs():
    for username in os.listdir(DEV_DIST_ROOT_DIR):
        print(f"- utilisateur {username} :")
        data_file = os.path.join(DEV_INFO % username)
        if not os.path.isfile(data_file):
            continue
        # supprimer les données
        devs = get_devs(username)
        for dev in devs:
            print(f'  suppression du devoir {dev}')
            supprimer(username, dev)
        os.unlink(data_file)
        try:
            print(f'  suppression du répertoire')
            os.rmdir(DEV_DIST_DIR % username)
        except:
            pass


################
# DISTRIBUTION #
def distribute(username, groupe, dev_name, eleve_only, in_perso=True):
    """distribue /home/workgroups/devoirs/<username>/<dev_name>/ aux membres de <groupe>
    """
    dev_name = validate_filename(dev_name)
    dev_name = dev_name.lower()
    if dev_name in RESERVED_NAMES:
        log.err('Nom du devoir invalide %s'%dev_name)
        return
    devs = get_devs(username)
    devs[dev_name] = {'groupe': groupe,
                          'eleve_only': eleve_only,
                          'state': 'err',
                          'timestamp': time.time()}
    if SUPPORT_ETAB:
        user = username
    else:
        user = None
    if eleve_only:
        user_liste = ldap_utils.get_membres_croise([groupe, 'eleves'], user=user)
    else:
        user_liste = ldap_utils.get_membres(groupe, user=user)
    # répertoire contenant le devoir chargé par Gestion-postes
    rep_path = os.path.join(DEV_DIST_DIR%username, dev_name)
    donnees = os.path.join(rep_path, 'donnees')
    # donner au élèves le droit de lire les données
#    if os.path.isdir(donnees): os.system('/usr/bin/setfacl -m g:%s:r-x %s'%(groupe, donnees))
    # distribution des fichiers
    reussi= 0
    rateu = []
    if in_perso:
        dev_dir = DEV_PERSO_BASE_DIR
    else:
        dev_dir = DEV_BASE_DIR
    for user in user_liste:
#        if 1:
        try:
            user_dev_dir = dev_dir%{'user': user}
            user_dev_sub_dir = os.path.join(user_dev_dir, username)
            if not os.path.exists(user_dev_sub_dir): os.makedirs(user_dev_sub_dir)
            user_dest_dir = os.path.join(user_dev_sub_dir, dev_name)
            copy_dev(rep_path, user_dest_dir, user)
            os.system('setfacl -PRm u:%s:rwx %s'%(user, user_dev_dir))
            os.system('setfacl -PRdm u:%s:rwx %s'%(user, user_dev_dir))
            reussi += 1
        except Exception as e:
            log.err(e)
            rateu.append(user)
    devs[dev_name]['state'] = 'dist'
    devs[dev_name]['in_perso'] = in_perso
    save_devs(username, devs)
    return (reussi, rateu, dev_name)


def get_document_user(user):
    if not REVERSE_LOGIN_DEVOIR:
        return user
    return get_reverse_login(user)


#############
# RAMASSAGE #
def ramasse(username, dev_name):
    dev_name = validate_filename(dev_name)
    devs = get_devs(username)
    dev = devs[dev_name]
    if SUPPORT_ETAB:
        user = username
    else:
        user = None
    if dev['eleve_only']:
        user_liste = ldap_utils.get_membres_croise([dev['groupe'], 'eleves'], user=user)
    else:
        user_liste = ldap_utils.get_membres(dev['groupe'], user=user)
    # ramasse les devoirs dans "perso\devoirs\rammasses"
    dest = os.path.join(DEV_PERSO_BASE_DIR%{'user':username}, 'ramasses', dev_name)
    if not os.path.exists(dest): os.makedirs(dest)
    # dans le perso de l'élève ?
    if devs[dev_name]['in_perso']:
        dev_dir = DEV_PERSO_BASE_DIR
    else:
        dev_dir = DEV_BASE_DIR
    reussi= 0
    rateu = []
    for user in user_liste:        # dossier <dev_dir>/
        user_dev_dir = dev_dir%{'user': user}
        # le dossier à ramasser "devoirs/<login_prof>/<nom du devoir>"
        src = os.path.join(user_dev_dir, username, dev_name)
        # accusé de ramassage
        accuse = os.path.join(src, '%s%s'%(dev_name, '.txt'))
        date = time.strftime("le %d/%m/%Y à %H:%M.", time.localtime())
        msg = 'Ce devoir a été ramassé %s'%date
        try:
            doc_user = get_document_user(user)
            for i in os.listdir(src):
                if i.lower() == 'donnees': continue
                f = os.path.join(src, i)
                # où le placer : "devoirs\ramasses\<nom du devoir>-<nom du groupe>"
                # en préfixant chaque élément : <user>--<nom du fichier original>
                dst = os.path.join(dest, "%s%s%s"%(doc_user, SEP_PREFIX, i))
                if os.path.exists(dst): shutil.rmtree(dst)
                p = os.path.split(dst)[0]
                if not os.path.isdir(p): os.makedirs(p)
                if os.path.isfile(f):
                    # on prévient le prof si la date de modification d'un fichier date de moins de 10 secondes après la date de création du devoir
                    # indispensable d'utiliser une fourchette de temps car la date du devoir est définie "un peu avant" la création des fichiers
                    if 'timestamp' in dev and (os.path.getmtime(f) - dev['timestamp'] < 10):
                        dst_non_modifie = os.path.join(dest, "%s%s%s%s"%(doc_user, SEP_PREFIX, FICHIER_NON_MODIFIE, i))
                        shutil.copy(f, dst_non_modifie)
                    else:
                        shutil.copy(f, dst)
                if os.path.isdir(f):
                    shutil.copytree(f, dst)
            reussi += 1
            open(accuse, 'w').write(msg)
            os.system ('setfacl -m u:%s:rwx "%s"'%(user, accuse))
        except Exception as e:
            rateu.append(user)
            log.err('Impossible de ramasser le devoir de %s : %s'%(user, e))
    os.system('setfacl -Rm u:%s:rwx %s'%(username, dest))
    os.system('setfacl -Rdm u:%s:rwx %s'%(username, dest))
    devs[dev_name]['state'] = 'ram'
    save_devs(username, devs)
    return (reussi, rateu)

###########
# RENDAGE #
def rendre(username, dev_name):
    """rend le devoir <dev_name> corrigé par <username> aux membres de <groupe>
    """
    dev_name = validate_filename(dev_name)
    # répertoire du devoir corrigé
    try:
        devs = get_devs(username)
        dev = devs[dev_name]
        source = os.path.join(DEV_PERSO_BASE_DIR%{'user':username}, 'ramasses', dev_name)
        reussi= 0
        rateu = []
        # récupération de la liste des destinataires du rendu
        destinataires = []
        for nom_dest in os.listdir(source):
            if SEP_PREFIX in nom_dest:
                destinataire = nom_dest.split(SEP_PREFIX)[0]
                doc_destinataire = get_document_user(destinataire)
                # vérifie que le destinataire est un utilisateur valide
                if len(destinataire) > 0 :
                    if os.path.isdir(os.path.join(PERSO_DIR%{'user': destinataire})):
                        destinataires.append(destinataire)
                    elif destinataire != doc_destinataire and os.path.isdir(os.path.join(PERSO_DIR%{'user': doc_destinataire})):
                        destinataires.append(doc_destinataire)
        # enlever les doublons de la liste
        destinataires = set(destinataires)

        for user in destinataires:
            # répertoire "U:\devoirs\<login.prof>\<dev_name>\correction" de l'élève
            user_dev_dir = DEV_PERSO_BASE_DIR%{'user': user}
            dst = os.path.join(user_dev_dir, username, dev_name, 'correction')
            if os.path.exists(dst): shutil.rmtree(dst)
            p = os.path.split(dst)[0]
            if not os.path.isdir(p): os.makedirs(p)
            try:
                os.mkdir(dst)
                # For compatibility raisons
                ori_user_with_sep = user + SEP_PREFIX
                doc_user = get_document_user(user)
                user_with_sep = doc_user + SEP_PREFIX
                for nom_fichier in os.listdir(source):
                    if (nom_fichier.startswith(user_with_sep) or nom_fichier.startswith(ori_user_with_sep)) and (len(nom_fichier) > len(user_with_sep)):
                        f = os.path.join(source, nom_fichier)
                        try:
                            # suppression du préfixe du nom de fichier et validation
                            nom_fichier_dest = validate_filename(nom_fichier.split(SEP_PREFIX, 1)[1])
                        except Exception as e:
                            log.err('Impossible de copier le fichier %s : %s'%(nom_fichier, e))
                            continue
                        f_dest = os.path.join(dst, nom_fichier_dest)
                        if os.path.isfile(f):
                            shutil.copyfile(f, f_dest)
                        if os.path.isdir(f):
                            shutil.copytree(f, f_dest)
                os.system('setfacl -PRm u:%s:rwx %s'%(user, dst))
                os.system('setfacl -PRdm u:%s:rwx %s'%(user, dst))
                reussi += 1
            except Exception as e:
                rateu.append(user)
                log.err('Impossible de rendre le devoir a %s : %s'%(user, e))
        devs[dev_name]['state'] = 'ren'
        save_devs(username, devs)
        return (reussi, rateu)
    except Exception as e:
        log.err('############ %s'%e)

###########################
# SUPPRESSION DES DONNEES #
def supprimer(username, dev_name):
    dev_name = validate_filename(dev_name)
    devs = get_devs(username)
    try:
        dev = devs[dev_name]
        # dans le perso ?
        if dev['in_perso']:
            dev_dir = DEV_PERSO_BASE_DIR
        else:
            dev_dir = DEV_BASE_DIR
        if dev['eleve_only']:
            user_liste = ldap_utils.get_membres_croise((dev['groupe'], 'eleves'))
        else:
            user_liste = ldap_utils.get_membres(dev['groupe'])
        for user in user_liste:
            user_dev_dir = DEV_BASE_DIR%{'user': user}
            cible = os.path.join(user_dev_dir, username, dev_name, 'donnees')
            try:
                if os.path.islink(cible): os.remove(cible)
            except:
                continue
    except: pass
    cible = os.path.join(DEV_DIST_DIR%username, dev_name)
    if os.path.isdir(cible): shutil.rmtree(cible)
    return True

def validate_filename(fname):
    valid_chars = "-_.%s%s" % (string.ascii_letters, string.digits)
    new_name = ""
    for i in fname:
        if i in valid_chars:
            new_name += i
    if len(new_name) == 0: raise Exception('Le nom ne peut pas etre nul ""')
    if new_name.count('.') == len(new_name): raise Exception('Le nom ne peut pas contenir que des point "."')
    return new_name
