# -*- coding: UTF-8 -*-

###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# twistd -noy controle_vnc_serveur.py
#
# Serveur Perspective Broker pour les clients Scribe/Windows
#
###########################################################################

# modules habituels
import os, sys
sys.path.append('/usr/share/eole/controlevnc')

# Perspective Broker
from twisted.spread import pb

# serveur HTTP
from twisted.web import server, static

# le service
from twisted.application import service, internet
from twisted.python import log
log.FileLogObserver.timeFormat = "%Y/%m/%d %H:%M:%S %Z"

# gestion de l'annuaire
import ldap_utils
from scribe.ldapconf import SUPPORT_ETAB
# gestion des clients
try:
    from cliscribe import Cliscribe
except:
    pass
from connexions import Connexions, Session
try:
    import gest_sessions

    # gestion des blocage réseau
    from blocage import Blocage
except:
    Blocage = None
    pass

# gestion des devoirs
from config import activer_eop, activer_eoe, debug
import devoirs

######################
#  appels distants   #
######################
class Echoer(pb.Root):
    """Classe contenant les fonctions distantes "remote_<func>" accessibles depuis les postes clients.
    Protection de certaines en fonction du groupe dont fait parti le login qui a ouvert une session Samba
    sur la station tentant d'accéder à la remote_fonction.
    """
    def __init__(self):
        if Blocage:
            self.blocage = Blocage()

#    def rootObject(self, broker):
#        """récupération d'informations sur la connection (ex: IP)
#        """
#        self.p_broker = broker
#        return pb.Root.rootObject(self, broker)

    def remoteMessageReceived(self, broker, message, args, kw):
        #self.ip = self.p_broker.transport.getPeer().host
        self.p_broker = broker
        self.ip = broker.transport.getPeer().host
        return pb.Root.remoteMessageReceived(self, broker, message, args, kw)

    #######################
    # Fonctions distantes #
    #######################

    def remote_bonjour(self, ip, ret=None):
        """Appel de la fonction bonjour sur client B par client A
        """
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_bonjour par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_bonjour par %s sur %s'%(self.ip, ip))
        d = Cliscribe(ip).appel()
        return d

    # Listes, des groupes, des connectés et des membres de groupes #
    def remote_connectes(self, groupes=None):
        """gestion controle-vnc
        """
        log.msg('Appel de la fonction remote_connectes (groupes=%s) par %s'%(groupes, self.ip))
        if groupes:
            if len(groupes) == 2 and groupes[1] == "eleves":
                return Connexions().get_connected_student(groupes[0])
            elif type(groupes) == str or type(groupes) == unicode:
                return Connexions().get_connected_by_group(groupes)
            else:
                raise Exception('Demande de groupe incorrecte')
        else:
            return Session().sessions

    def remote_groupes(self):
        """renvoie la liste des groupes
        """
        log.msg('Appel de la fonction remote_groupes par %s'%self.ip)
        if SUPPORT_ETAB:
            user = Connexions().get_user(self.ip)
        else:
            user = None
        return ldap_utils.get_groupes(user=user)

    def remote_classes(self):
        """renvoie la liste des classes
        """
        log.msg('Appel de la fonction remote_classes par %s'%self.ip)
        return ldap_utils.get_classes()

    def remote_classes_et_groupes(self):
        """renvoie la liste des classes et des groupes de type "Groupe"
        """
        log.msg('Appel de la fonction remote_classes_et_groupes par %s'%self.ip)
        if SUPPORT_ETAB:
            user = Connexions().get_user(self.ip)
        else:
            user = None
        return ldap_utils.get_classes_et_groupes(user=user)

    def remote_membres(self, groupe, rgroupe=None):
        """renvoie la liste des membres de 'groupe'
        """
        log.msg('Appel de la fonction remote_membres par %s (%s, %s)'%(self.ip, groupe, rgroupe))
        if rgroupe:
            return ldap_utils.get_membres_croise([groupe, rgroupe])
        else:
            return ldap_utils.get_membres(groupe)

    # Observation VNC #
    def remote_get_vnc_mode(self):
        log.msg('Appel de la fonction remote_get_vnc_mode par %s'%(self.ip))
        return gest_sessions.Logon().get_vnc_mode()

    def remote_observe(self, cible, active=None):
        """src est en mode "listen"
        clibe "connect" src, sur cible : 'winvnc.exe -connect src (self.ip)'
        """
        connexions = Connexions()
        if not connexions.isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_observe par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_observe par %s sur %s (active = %s)'%(self.ip, cible, active))
        # force le mode d'observation fixé dans l'EAD
        if not connexions.isadmin(self.ip) and gest_sessions.Logon().get_vnc_mode() == 'simple':
            active=None
        d = Cliscribe(cible).vnc(action='setinputs', value=active, restart=True)
        d.addCallback(Cliscribe(cible).vnc, action='connect', value=self.ip)
        return d

    # Diffusion VNC #
    def remote_diffusion(self, ip_liste):
        """active le mode vncviewer /listen sur les ips de ip_liste
        """
        if not ip_liste:
            return
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_diffusion par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_diffusion par %s sur %s'%(self.ip, ip_liste))
        # fichier généré par vncviewer.exe configuration en 256 couleurs
        myconf = file('/home/workgroups/professeurs/gestion-postes/eole_conf.vnc').read()
        # winvnc en mode lecture seule (fait dans la session)
        #d = Cliscribe(self.ip).vnc(action='setinputs', value=None)
        d = Cliscribe(ip_liste[0]).vnc(action='start_listen', value='/viewonly', conf=myconf)
        for cible in ip_liste[1:]:
            d.addCallback(Cliscribe(cible).vnc, action='start_listen', value='/viewonly', conf=myconf)
            d.addErrback(Cliscribe(cible).vnc, action='start_listen', value='/viewonly', conf=myconf)
        return d

    # Blocage#
    def remote_get_bloc_list(self):
        """Renvoie la liste des type de blocages disponibles (Aucun=0, Tout internet=1, ...)
        """
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_get_bloc_list par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_get_bloc_list par %s'%self.ip)
        return self.blocage.get_bloc_list()

    # ... par Machine
    def remote_get_machines_blocmod(self):
        """renvoie la liste des machines avec leur blocage actuel
        """
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_get_machines_blocmod par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_get_machines_blocmod par %s'%self.ip)
        return self.blocage.get_computers_mod()

    def remote_set_machine_bloc_mod(self, machines):
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_set_machine_bloc_mod par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_set_machine_bloc_mod par %s, Machines=%s'%(self.ip, machines))
        return self.blocage.bloc_computers(machines)

    # ... par Utilisateur
    def remote_users_bloc_mod(self, groupe):
        """renvoie un dictionnaire des membres du groupe avec leur mode de blocage
        { user1 : {'netmod' : 1, 'partmod' : 0}, user2 :  ... }
        """
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_users_bloc_mod par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_users_bloc_mod par %s GRP=%s'%(self.ip, groupe))
        if SUPPORT_ETAB:
            user = Connexions().get_user(self.ip)
        else:
            user = None
        return self.blocage.get_users_mod(groupe, user=user)

    def remote_set_user_bloc_mod(self, users):
        """Applique le blocage demandé aux utilisateurs sélectionnés
        users = [(user, netmod, partmod, durée du blocage), (user2, ...)]
        """
        if not Connexions().isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_set_users_bloc_mod par %s'%self.ip)
            return
        log.msg('Appel de la fonction remote_set_user_bloc_mod par %s, Users=%s'%(self.ip, users))
        return self.blocage.bloc_users(users)

    # Distribution de devoirs
    def remote_nb_membres(self, groupe):
        """Renvoie le nombre de membres du groupe
        """
        log.msg('Appel de la fonction remote_nb_membres par %s pour %s'%(self.ip, groupe))
        if SUPPORT_ETAB:
            user = Connexions().get_user(self.ip)
        else:
            user = None
        return ldap_utils.get_nb_membres(groupe, user=user)

    def remote_get_devoirs(self, user=None):
        connexions = Connexions()
        if not connexions.isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_get_devoirs par %s'%self.ip)
            return
        if not user:
            user = connexions.get_user(self.ip)
        log.msg('Appel de la fonction remote_get_devoirs par %s : user=%s' % \
                (self.ip, user))
        return devoirs.get_devs(user)

    def remote_distribute(self, groupe, dev_name, eleve_only, in_perso, user=None):
        """distribue le devoir "dev_name" du prof "username" à "groupe"
        """
        connexions = Connexions()
        if not connexions.isprof(self.ip):
             self.p_broker.transport.loseConnection()
             log.msg('Appel non autorisé (non prof) de remote_distribute par %s'%self.ip)
             return
        if not user:
            user = connexions.get_user(self.ip)
        log.msg('Appel de la fonction remote_distribute par %s : user=%s, groupe=%s, dev_name=%s, eleve_only=%s, %s'%(self.ip, user, groupe, dev_name, eleve_only, in_perso))
        return devoirs.distribute(user, groupe, dev_name, eleve_only, in_perso)

    def remote_ramasse(self, dev_name, user=None):
        """rammasse "dev_name"
        """
        connexions = Connexions()
        if not connexions.isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_ramasse par %s'%self.ip)
            return
        if not user:
            user = connexions.get_user(self.ip)
        log.msg('Appel de la fonction remote_ramasse par %s : user=%s, dev_name=%s'%(self.ip, user, dev_name))
        return devoirs.ramasse(user, dev_name)

    def remote_rendre(self, dev_name, user=None):
        """rend la correction de "dev_name"
        """
        connexions = Connexions()
        if not connexions.isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_rendre par %s'%self.ip)
            return
        if not user:
            user = connexions.get_user(self.ip)
        log.msg('Appel de la fonction remote_rendre par %s : user=%s, dev_name=%s'%(self.ip, user, dev_name))
        return devoirs.rendre(user, dev_name)

    def remote_supprimer(self, dev_name, user=None):
        """Supprime dev_name de la liste des devoirs et supprime les données correspondantes
        (sinon l'élève ne peut supprimer le dossier du devoir dans son rep perso
        """
        connexions = Connexions()
        if not connexions.isprof(self.ip):
            self.p_broker.transport.loseConnection()
            log.msg('Appel non autorisé (non prof) de remote_supprimer par %s'%self.ip)
            return
        if not user:
            user = connexions.get_user(self.ip)
        log.msg('Appel de la fonction remote_supprimer par %s : user=%s, dev_name=%s'%(self.ip, user, dev_name))
        return devoirs.supprimer(user, dev_name)

    # Gestion des postes (démarrage) #
    def remote_service_start(self, nom, ost=None, mac=None):
        """gestion des postes allumés
        """
        log.msg('Appel de la fonction remote_service_start par %s : %s (%s), mac=%s'%(self.ip, nom, ost, mac))
        return Connexions().service_start(self.ip, mac)

    # Gestion des postes (arrêt) #
    def remote_service_stop(self):
        log.msg('Appel de la fonction remote_service_stop par %s'%self.ip)
        return Connexions().service_stop(self.ip)

    # Gestion de sessions cliente #
    # Ouverture de session
    def remote_logon(self, os_type):
        """fonction appelée par les clients lors de l'ouverture de session
        (dés)active le blocage
        """
        log.msg('Appel de la fonction remote_logon par %s'%(self.ip))
        return gest_sessions.Logon().logon(self.ip, os_type, self.blocage)


#######################
# programme principal #
#######################
if __name__ == '__main__':
    sys.exit("Usage : twistd -noy %s"%os.path.basename(sys.argv[0]))

# Après un arrêt de controle-vnc, purge des stations allumees
application = service.Application('ControleVNCserveur')
# serveur de téléchargement de client Scribe
root = static.File('/home/client_scribe')
site = server.Site(root)
log.msg('Controle VNC Serveur - Fichiers demarre port:8790')
internet.TCPServer(8790, site).setServiceParent(service.IServiceCollection(application))
# serveur PB pour exécution de fonctions distantes
pbfactory = pb.PBServerFactory(Echoer(), unsafeTracebacks=True)
#pbfactory = pb.PBServerFactory(Echoer(), unsafeTracebacks=False)
log.msg('Controle VNC Serveur demarre port:8789')
internet.TCPServer(8789, pbfactory).setServiceParent(service.IServiceCollection(application))
#PBServerFactory.unsafeTracebacks

# serveur xmlrpc pour EOP (gestion de devoirs)
if activer_eop == 'oui':
    from xmlrpc_eop import RpcServer
    from json import load

    # chargement de la clé secrète pour les appels xmlrpc
    try:
        with open("/etc/eole/flask/keys/eop.key", "r") as keyfile:
            content = load(keyfile)
            secret_key = content['secret_key']
    except:
        log.msg('Impossible de lire le fichier /etc/eole/flask/keys/eop.key nécessaire pour EOP, la gestion des devoirs')
        secret_key=""

    rpcserver = server.Site(RpcServer(secret_key))
    log.msg('Controle VNC Serveur demarre port:8788')
    internet.TCPServer(8788, rpcserver).setServiceParent(service.IServiceCollection(application))

if activer_eoe == 'oui':
    from xmlrpc_eoe import RpcServer
    from json import load

    # chargement de la clé secrète pour les appels xmlrpc
    try:
        with open("/etc/eole/flask/keys/eoe.key", "r") as keyfile:
            content = load(keyfile)
            secret_key = content['secret_key']
    except:
        log.msg('Impossible de lire le fichier /etc/eole/flask/keys/eoe.key nécessaire pour EOE, les outils élèves')
        secret_key=""

    rpcserver = server.Site(RpcServer(secret_key))
    log.msg('Controle VNC Serveur demarre port:8792')
    internet.TCPServer(8792, rpcserver).setServiceParent(service.IServiceCollection(application))
