<?php

namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Id\AssignedGenerator;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

use App\Service\toolService;

use App\Entity\Mindmap as Entity;
use App\Form\MindmapType as Form;

class MindmapController extends AbstractController
{
    private $data   = "mindmap";
    private $route  = "app_user_mindmap";
    private $render = "Mindmap/";
    private $entity = "App:Mindmap";
    private $appKernel;
    private $tool;
    
    public function __construct(KernelInterface $appKernel,toolService $tool)
    {
        $this->appKernel = $appKernel;
        $this->tool = $tool;
    }

    public function list(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $datas = $em->getRepository($this->entity)->findAll();

        return $this->render($this->render.'list.html.twig',[
            $this->data."s" => $datas,
            "useheader"     => true,
            "usesidebar"    => true,
        ]);
    }

    public function view($id, Request $request, $access="user")
    {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');

        $perm = $this->perm($data);
        if(!$perm) return $this->redirect($this->generateUrl('app_user_noperm',["displayname"=>$data->getUser()->getDisplayname(),"email"=>$data->getUser()->getEmail(),"url"=>$this->generateUrl($this->route."_view",["id"=>$id], UrlGeneratorInterface::ABSOLUTE_URL)]));    

        $mindmapscale=$em->getRepository("App:User")->getUserpreference($this->getUser(),"mindmapscale",$id);
        if(!$mindmapscale) $mindmapscale=1;
        
        $form = $this->createForm(Form::class,$data,array("mode"=>"view", "access" => $access));

        return $this->render($this->render.'view.html.twig',[
            'useheader'     => true,
            'usemenu'       => false,
            'usesidebar'    => ($access=="admin"),   
            'maxwidth'      => false,           
            $this->data     => $data,
            'form'          => $form->createView(),
            'access'        => $access,
            'mindmapscale'  => $mindmapscale,
            'perm'          => $perm,
        ]);
    }

    public function info($id, Request $request, $access="user")
    {
        // S'assurer que c'est un appel ajax
        if (!$request->isXmlHttpRequest()) {
            return new JsonResponse(array('message' => 'Interdit'), 400);
        }

        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) return new JsonResponse(array('message' => 'Interdit'), 400);

        $output=[
            "id" => $data->getId(),
            "title" => $data->getTitle(),
            "description" => $data->getDescription(),
        ];

        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');      
        return $response;
    }

    public function submit(Request $request, $access="user") {
        // Initialisation de l'enregistrement
        $em = $this->getDoctrine()->getManager();
        $data = new Entity();
        if($access=="user") $data->setUser($this->getUser());
        $data->setDescription('<mxGraphModel><root><mxCell id="0"/><mxCell id="1" parent="0"/><cellnode level="0" label="Idée" id="2"><mxCell style="fillColor=#2574a9;strokeColor=#00002a;fontSize=35;fontColor=#ffffff" parent="1" vertex="1"><mxGeometry x="-200" y="2320" width="280" height="270" as="geometry"/></mxCell></cellnode></root></mxGraphModel>');

        // Création du formulaire
        $form = $this->createForm(Form::class,$data,array("mode"=>"submit", "access" => $access));

        // Récupération des data du formulaire
        $form->handleRequest($request);
        
        // Sur erreur
        $this->getErrorForm(null,$form,$request,$data,"submit");
        
        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            $data = $form->getData();  
            
            $em->persist($data);
            $em->flush();

            // Retour à la liste
            return $this->redirect($this->generateUrl('app_'.$access.'_mindmap_view',["id"=>$data->getId()]));    
        }

        // Affichage du formulaire
        return $this->render($this->render.'edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => ($access=="admin"),     
            'maxwidth'          => ($access=="user"),            
            $this->data         => $data,
            'mode'              => 'submit',
            'form'              => $form->createView(),
            'access'            => $access,
        ]);    
    }

    public function update($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');
        if($data->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO")) 
        throw $this->createNotFoundException('Permission denied');

        // Création du formulaire
        $form = $this->createForm(Form::class,$data,array("mode"=>"update", "access" => $access));

        // Récupération des data du formulaire
        $form->handleRequest($request);

        // Sur erreur
        $this->getErrorForm(null,$form,$request,$data,"update");
        
        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            $data = $form->getData();  
            $em->persist($data);
            $em->flush();

            // Retour à la liste
            if($access=="user")
                return $this->redirect($this->generateUrl('app_'.$access.'_mindmap_view',["id"=>$data->getId()]));          
            else 
                return $this->redirect($this->generateUrl('app_admin_mindmap'));     

        }
        
        // Affichage du formulaire
        return $this->render($this->render.'edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => ($access=="admin"),  
            'maxwidth'          => ($access=="user"),
            $this->data         => $data,
            'mode'              => 'update',
            'form'              => $form->createView(),
            'access'            => $access
        ]);         
    }

    public function delete($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');
        if($data->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO"))  
        throw $this->createNotFoundException('Permission denied');

        $em->remove($data);
        $em->flush();

        if($access=="user")
            return $this->redirect($this->generateUrl('app_home'));          
        else
            return $this->redirect($this->generateUrl('app_admin_mindmap'));          
    }    

    public function export($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $mindmap=$em->getRepository($this->entity)->find($id);
        if (!$mindmap) throw $this->createNotFoundException('Unable to find entity.');
        if($mindmap->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO")) 
        throw $this->createNotFoundException('Permission denied');

        $fs = new Filesystem();
        $rootdir = $this->appKernel->getProjectDir();
        $destdir = $rootdir."/uploads/export/mindmap/$id";

        // Regénération du répertoire d'export
        $fs->remove($destdir);
        $fs->mkdir($rootdir."/uploads");
        $fs->mkdir($rootdir."/uploads/export");
        $fs->mkdir($rootdir."/uploads/export/mindmap");
        $fs->mkdir($rootdir."/uploads/export/mindmap/$id");

        // Création du json d'export
        $export = new \stdClass();
        $export->type = "mindmap";
        $export->id = $mindmap->getId();
        $export->title = $mindmap->getTitle();
        $export->description = $mindmap->getDescription();
        $export->user = $mindmap->getUser()->getId();

        $export->groups = [];
        $groups=$mindmap->getGroups();
        foreach($groups as $group) {
            $idgrp=$group->getId();
            $export->groups[$idgrp] = new \stdClass();
            $export->groups[$idgrp]->id=$idgrp;
        }
        
        $export->groupreaders = [];
        $groups=$mindmap->getGroupreaders();
        foreach($groups as $group) {
            $idgrp=$group->getId();
            $export->groupreaders[$idgrp] = new \stdClass();
            $export->groupreaders[$idgrp]->id=$idgrp;
        } 

        $export->users = [];
        $users=$mindmap->getUsers();
        foreach($users as $user) {
            $idusr=$user->getId();
            $export->users[$idusr] = new \stdClass();
            $export->users[$idusr]->id=$idusr;
        }
        
        $export->userreaders = [];
        $users=$mindmap->getUserreaders();
        foreach($users as $user) {
            $idusr=$user->getId();
            $export->userreaders[$idusr] = new \stdClass();
            $export->userreaders[$idusr]->id=$idusr;
        } 

        file_put_contents($destdir."/info.json",json_encode($export));
        $now=new \Datetime();
        $zipName=$rootdir."/uploads/export/Cartementale-".$now->format("Ymd")."-$id.zip";
        $this->tool->zip($destdir,$zipName);

        $response = new BinaryFileResponse($zipName);
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_INLINE);
        return $response;
    }


    public function import(Request $request, $access="user") {
        return $this->render($this->render.'import.html.twig',[
            'useheader'     => true,
            'usemenu'       => false,
            'usesidebar'    => ($access=="admin"),
            'access'        => $access
        ]);
    }

    public function importzip($access="user")
    {
        return $this->render($this->render.'importzip.html.twig',[
            'useheader'     => false,
            'usemenu'       => false,
            'usesidebar'    => false,
            'access'        => $access
        ]);
    }
    
    public function importexec(Request $request,$access="user")
    {
        if (!$request->isXmlHttpRequest()) return new JsonResponse(array('message' => 'Interdit'), 400);

        $em = $this->getDoctrine()->getManager();
        $metadata =  $em->getClassMetaData($this->entity);
        $metadata->setIdGeneratorType(ClassMetadata::GENERATOR_TYPE_NONE);
        $metadata->setIdGenerator(new AssignedGenerator());

        $output=array();

        // Décompresser le fichier zip et controler son format
        $rootdir = $this->appKernel->getProjectDir();
        $zip = $rootdir."/".$request->request->get('file');
        $tmp=explode("/",$zip);
        $tmp=explode(".",end($tmp));
        $id=$tmp[0];
        $destdir = $rootdir."/uploads/import/$id";
        $unzipped = $this->tool->unzip($zip,$destdir);
        $error="";
        if(!$unzipped) 
            return $this->importreturnerror("<p>Votre fichier ZIP n'a pas pu être décompressé<p>");
        else {
            // Chargement du fichier json associé
            $json=file_get_contents($destdir."/info.json");
            if(!$json) return $this->importreturnerror("<p>Contenu du fichier ZIP invalide<p>");;
            $json= json_decode($json);
            if(!$json) return $this->importreturnerror("<p>Contenu du fichier ZIP invalide<p>");;
        }

        // Structure attendu
        $frmboards=["type","id","title","description","user","groups","groupreaders","users","userreaders"];

        // On vérifie la structure du json
        foreach($frmboards as $frm) {
            if (!property_exists($json,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété $frm manquante<p>";
        }
        if($error) return $this->importreturnerror($error);
        if($json->type!="mindmap") return $this->importreturnerror("<p>Contenu du fichier ZIP invalide = type invalide<p>");

        // On s'assure que le board à restaurer existe
        $board=$em->getRepository($this->entity)->find($json->id);
        if(!$board) return $this->importreturnerror("<p>Impossible de restaurer le board n'existe pas</p>");      

        // On s'assure que l'utilisateur à la permission de restaurer ce board
        if($access=="user") {
            if($board->getUser()!=$this->getUser()) return $this->importreturnerror("<p>N'étant pas propriétaire du board vous ne pouvait pas le restaurer</p>");
        }
        $owner=$board->getUser();

        if($error!="") {
            $output["status"]="KO";
            $output["error"]=$error;
        }
        else {
            // On supprime le board pour le regénrer
            $em->remove($board);
            $em->flush();
               
            // Génération de la sauvegarde
            $board = new Entity();
            $board->setId($json->id);
            $board->setTitle($json->title);
            $board->setDescription($json->description);
            $board->setUser($owner);

            foreach($json->groups as $widgrp) {
                $group=$em->getRepository("App:Group")->find($widgrp->id);
                if($group) $board->addGroup($group);
            }
            foreach($json->groupreaders as $widgrp) {
                $group=$em->getRepository("App:Group")->find($widgrp->id);
                if($group) $board->addGroupreader($group);
            }
            
            foreach($json->users as $widusr) {
                $user=$em->getRepository("App:User")->find($widusr->id);
                if($user) $board->addUser($user);
            }
            foreach($json->userreaders as $widusr) {
                $user=$em->getRepository("App:User")->find($widusr->id);
                if($user) $board->addUserreader($user);
            }

            $em->persist($board);
            $em->flush();
        }

        $output["status"]="OK";
        $output["id"]=$json->id;
        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');        
        return $response;   
    }

    private function importreturnerror($error) {
        $output["status"]="KO";
        $output["error"]=$error;

        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');        
        return $response;        
    }

    private function perm($entity) {
        $perm=false;

        if($entity->getUser()==$this->getUser()) $perm="write";
        if($this->getUser()->hasRole("ROLE_ADMIN")) $perm="write";
        if($this->getUser()->hasRole("ROLE_MODO")) $perm="write";

        // Récupération des groupes de l'utilisateur
        $groups=$this->getUser()->getGroups();

        // Ses groupes ont-ils la permission 
        foreach($groups as $group) {
            if($entity->getGroups()->contains($group)) $perm="write";
        }

        // Son compte a-til la permission
        if($entity->getUsers()->contains($this->getUser())) $perm="write";

        // Si pas de permission on regarde s'il a les permissions en lecture
        if(!$perm) {
            // Ses groupes ont-ils la permission 
            foreach($groups as $group) {
                if($entity->getGroupreaders()->contains($group)) $perm="read";
            }

            // Son compte a-til la permission
            if($entity->getUserreaders()->contains($this->getUser())) $perm="read";
        }

        // Erreur si non permis
        if(!$perm) return false;
        
        return $perm;
    }

    protected function getErrorForm($id,$form,$request,$data,$mode) {
        if ($form->get('submit')->isClicked()&&$mode=="delete") {
        }

        if ($form->get('submit')->isClicked() && $mode=="submit") {
        }

        if ($form->get('submit')->isClicked() && ($mode=="submit" || $mode=="update")) {
        }

        if ($form->get('submit')->isClicked() && !$form->isValid()) {
            $this->get('session')->getFlashBag()->clear();

            $errors = $form->getErrors();
            foreach( $errors as $error ) {
                $request->getSession()->getFlashBag()->add("error", $error->getMessage());
            }
        }
    }      
}
