<?php
namespace App\Websocket;
 
use Exception;
use Ratchet\ConnectionInterface;
use Ratchet\MessageComponentInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Doctrine\ORM\EntityManagerInterface;
use SplObjectStorage;
use Symfony\Component\Filesystem\Filesystem;

use App\Entity\Scrum;
use App\Entity\Scrumcolumn;
use App\Entity\Scrumwidget;

use App\Entity\Whiteboard;
use App\Entity\Whiteboardwidget;

use App\Entity\Wordcloud;
use App\Entity\Wordcloudword;

use App\Entity\Timeline;
use App\Entity\Timelinesegment;
use App\Entity\Timelinedate;

use App\Entity\Message;
use App\Entity\User;

const BLACK_COLOR = '#000000';

class MessageHandler implements MessageComponentInterface
{
    protected $container;
    protected $em;

    protected $clients;
    private $channels;
    private $channeltypes;
    private $channelkeys;
    private $userconns;
    private $userkeys;
    private $fgdebug;
    private $filesystem;
 
    public function __construct(ContainerInterface $container, EntityManagerInterface $em)
    {
        $this->container = $container;
        $this->em = $em;

        $this->clients          = new SplObjectStorage;
        $this->channels         = [];
        $this->channeltypes     = [];
        $this->channelkeys      = [];
        $this->userconns        = [];
        $this->userkeys         = [];
        $this->filesystem       = new Filesystem();
        $this->fgdebug          = true;
    }

    private function debug($string) { 
        if($this->fgdebug) {
            $this->filesystem->appendToFile('/var/www/html/nineboard/var/log/wssdebug.log', $string."\n");
        }
    }      
 
    public function onOpen(ConnectionInterface $conn)
    {
        $this->clients->attach($conn);
        $this->userconns[$conn->resourceId] = $conn;
    }

    public function onClose(ConnectionInterface $conn)
    {
        $data= new \stdClass;
        $data->command = "adead";

        // Envoyer le message de deconnection
        if (isset($this->channels[$conn->resourceId])) {
            $this->sendMessage($conn,$data,"other");
        }        
        
        // Détacher
        $this->clients->detach($conn);
        unset($this->userconns[$conn->resourceId]);
        unset($this->userkeys[$conn->resourceId]);
        unset($this->channels[$conn->resourceId]);
    }

    public function onError(ConnectionInterface $conn, Exception $e)
    {
        $conn->close();
    }

    public function onMessage(ConnectionInterface $conn, $msg)
    {
        $data = json_decode($msg);

        switch ($data->command) {
            case "subscribe":
                $this->channels[$conn->resourceId] = $data->channel;
                $this->channeltypes[$conn->resourceId] = $data->channeltype;
                $this->channelkeys[$conn->resourceId] = $data->channelkey;
                $this->userkeys[$conn->resourceId] = $data->userkey;
            break;

            case "meto":
                if (isset($this->channels[$conn->resourceId])) {
                    $this->sendMessage($conn,$data,"other");
                }
            break;

            case "alive":
                if (isset($this->channels[$conn->resourceId])) {
                    $this->sendMessage($conn,$data);
                }
            break;

            default:
                if (isset($this->channels[$conn->resourceId])) {
                    switch ($this->channeltypes[$conn->resourceId]) {
                        case "scrum": $this->onMessageScrum($conn,$data); break;
                        case "whiteboard": $this->onMessageWhiteboard($conn,$data); break;
                        case "wordcloud": $this->onMessageWordcloud($conn,$data); break;
                        case "timeline": $this->onMessageTimeline($conn,$data); break;
                        case "mindmap": $this->onMessageMindmap($conn,$data); break;
                        default: $this->sendMessage($conn,$data);
                    }
                }
            break;
        }
    } 

    private function onMessageScrum(ConnectionInterface $conn, $data) {
        $id=$this->channelkeys[$conn->resourceId];

        switch ($data->command) {

            case "unnotify":
                $this->sendMessage($conn,$data);
            break;

            case "addcol":        
                // Récupérer le scrum
                $scrum=$this->em->getRepository("App:Scrum")->find($id);

                
                // On resize les colonnes
                $scrumcolumns=$this->em->getRepository("App:Scrumcolumn")->findBy(['scrum'=>$scrum],['roworder'=>'ASC']);
                $nbcol=0;
                foreach($scrumcolumns as $scrumcolumn) {
                    $scrumcolumn->setSize(str_replace("col-md-","",$data->sizes[$nbcol]));
                    $nbcol++;
                }

                
                // On recherche le derniers roworder de colonne pour ce scrum
                $last=$this->em->getRepository("App:Scrumcolumn")->findOneBy(['scrum'=>$scrum],['roworder'=>'DESC']);
                $roworder=($last?$last->getRoworder()+1:1);

                
                // On gérère la nouvelle colonne
                $col= new Scrumcolumn();
                $col->setTitle("COLONNE ".$roworder);
                $col->setRoworder($roworder);
                $col->setScrum($scrum);
                $col->setSize(str_replace("col-md-","",$data->sizes[$nbcol]));
                $this->em->persist($col);
                $this->em->flush();

                // Retour
                $data->id = $col->getId();
                $data->title= $col->getTitle();
                
                $this->sendMessage($conn,$data);
            break;

            case "delcol":     
                // Récupérer le scrum
                $scrum=$this->em->getRepository("App:Scrum")->find($id);
    
                // On recherche la colonne pour la supprimer
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                if($col) {
                    $this->em->remove($col);
                    $this->em->flush(); 
                }
    
                // On resize les colonnes
                $scrumcolumns=$this->em->getRepository("App:Scrumcolumn")->findBy(['scrum'=>$scrum],['roworder'=>'ASC']);
                $nbcol=0;
                foreach($scrumcolumns as $scrumcolumn) {
                    $scrumcolumn->setSize(str_replace("col-md-","",$data->sizes[$nbcol]));
                    $nbcol++;
                    $this->em->flush(); 
                }
                                
    
                // Send event
                $this->sendMessage($conn,$data);
            break;


            case "maxleftcol":   
                // On resize la colonne en cours
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                if($col) {
                    $col->setSize($data->size);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // On resize la colonne précédente
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->idprev);
                if($col) {
                    $col->setSize($data->sizeprev);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;
            
            case "maxrightcol":   
                // On resize la colonne en cours
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                if($col) {
                    $col->setSize($data->size);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // On resize la colonne suivante
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->idnext);
                if($col) {
                    $col->setSize($data->sizenext);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // Send event            
                $this->sendMessage($conn,$data);
            break;            

            case "startmovecol":
                // Send event            
                $data->message= "déplace la colonne";
                $this->sendMessage($conn,$data);
            break;          
    
            case "ordercol":   
                // On ordonne la colonne en cours
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                if($col) {
                    $col->setRoworder($data->order);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // Send event            
                $this->sendMessage($conn,$data);
            break;                

            case "starteditcol":
                $data->message= "modifie le titre de la colonne";
                $this->sendMessage($conn,$data);
            break;

            case "editcol" :
                // On modifie le titre de la colonne en cours
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                if($col) {
                    $col->setTitle($data->title);
                    $this->em->persist($col);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;

            case "addwid" :
                // Récupérer la colonne
                $scrumcolumn=$this->em->getRepository("App:Scrumcolumn")->find($data->id);
                
                // Récupérer le créateur
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                
                // On recherche le derniers roworder de widget pour cette colonne
                $last=$this->em->getRepository("App:Scrumwidget")->findOneBy(['scrumcolumn'=>$scrumcolumn],['roworder'=>'DESC']);
                $roworder=($last?$last->getRoworder()+1:1);
    
    
                // On gérère le nouveau widget
                $wid= new Scrumwidget();
                $wid->setTitle("BILLET ".$roworder);
                $wid->setRoworder($roworder);
                $wid->setScrumcolumn($scrumcolumn);
                $wid->setUser($user);

                $this->em->persist($wid);
                $this->em->flush();
    
                // Send event
                $data->idwid = $wid->getId();
                $data->title = $wid->getTitle();
                $data->description = "";
                $data->username = ($user?$user->getUsername():"");
                $data->filetype = $wid->getFiletype();
                $data->file = $wid->getFile();

                // Modification de l'url youtube
                if($data->filetype=="youtube"&&stripos($data->file,"https://www.youtube.com/embed")===false&&stripos($data->file,"youtube")!==false) {
                    $data->file=str_replace("http://www.youtube.com","https://www.youtube.com",$data->file);
                    $data->file=str_replace("https://www.youtube.com/watch?v=","",$data->file);    
                    $tmp=explode("&",$data->file);
                    $data->file="https://www.youtube.com/embed/".$tmp[0];                                         
                }

                // Modification de l'url peertube
                if($data->filetype=="youtube"&&stripos($data->file,"/watch/")!==false) {
                    $data->file=str_replace("/watch/","/embed/",$data->file);    
                }

                $this->sendMessage($conn,$data);
            break;   
            
            case "delwid" :
                // On recherche le widget pour le supprimer
                $wid=$this->em->getRepository("App:Scrumwidget")->find($data->id);
                if($wid) {
                    $this->em->remove($wid);
                    $this->em->flush(); 
                }                         
    
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            

            case "startmovewid" :
                // Send event
                $data->message= "déplace le billet";
                $this->sendMessage($conn,$data);
            break;
            
            case "orderwid" :
                // On recherche la colonne liée au widget
                $col=$this->em->getRepository("App:Scrumcolumn")->find($data->colid);
                if($col) {
                    // On ordonne la colonne en cours
                    $wid=$this->em->getRepository("App:Scrumwidget")->find($data->id);
                    if($wid) {
                        $wid->setRoworder($data->order);
                        $wid->setScrumcolumn($col);
                        $this->em->persist($wid);
                        $this->em->flush();                
                    }
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;

            case "starteditwid" :
                // Send event
                $data->message= "modifie le billet";
                $this->sendMessage($conn,$data);
            break;

            case "editwid" :
                // On modifie le widget en cours
                $wid=$this->em->getRepository("App:Scrumwidget")->find($data->id);
                if($wid) {
                    // Modification de l'url youtube
                    if($data->filetype=="youtube"&&stripos($data->file,"https://www.youtube.com/embed")===false&&stripos($data->file,"youtube")!==false) {
                        $data->file=str_replace("http://www.youtube.com","https://www.youtube.com",$data->file);
                        $data->file=str_replace("https://www.youtube.com/watch?v=","",$data->file);     
                        $tmp=explode("&",$data->file);
                        $data->file="https://www.youtube.com/embed/".$tmp[0];                                       
                    }

                    // Modification de l'url peertube
                    if($data->filetype=="youtube"&&stripos($data->file,"/watch/")!==false) {
                        $data->file=str_replace("/watch/","/embed/",$data->file);    
                    }

                    $wid->setTitle($data->title);
                    $wid->setDescription($data->description);
                    $wid->setFiletype($data->filetype);
                    $wid->setFile($data->file);
                    $this->em->persist($wid);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;
    
            case "canceleditwid" :
                // Send event
                $this->sendMessage($conn,$data);
            break;

            case "startuplowid" :
                // Send event
                $data->message= "télécharge des PJ";
                $this->sendMessage($conn,$data);
            break;  
            
            case "stopuplowid" :
                // Send event
                $this->sendMessage($conn,$data);
            break;   
            
            case "addmessage" :
                $scrum=$this->em->getRepository("App:Scrum")->find($id);
                $wid=$this->em->getRepository("App:Scrumwidget")->find($data->id);
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                if($wid&&$user) {
                    // On gérère le nouveau message
                    $mes= new Message();
                    $mes->setMessage($data->message);
                    $mes->setScrumwidget($wid);
                    $mes->setUser($user);

                    $this->em->persist($mes);
                    $this->em->flush();      
                    
                    $data->idmsg=$mes->getId();
                    $data->date=$mes->getSubmitdate()->format("d/m/Y H:i");
                    $data->usernamne=$mes->getUser()->getUsername();
                    $data->fgcandel=false;
                    $data->usermsg=$mes->getUser()->getId();
                    $data->userwid=($wid->getUser()?$wid->getUser()->getId():$scrum->getUser()->getId());
                    $data->userboa=$scrum->getUser()->getId();
                }
                $this->sendMessage($conn,$data);
            break;

            case "delmessage" :
                $msg=$this->em->getRepository("App:Message")->find($data->id);
                if($msg) {
                    $data->idwid=$msg->getScrumwidget()->getId();
                    $this->em->remove($msg);
                    $this->em->flush(); 
                }

                $this->sendMessage($conn,$data);
            break;            
        }
    }


    private function onMessageWhiteboard(ConnectionInterface $conn, $data) {
        $id=$this->channelkeys[$conn->resourceId];

        switch ($data->command) {

            case "unnotify":
                $this->sendMessage($conn,$data);
            break;

            case "addwid" :
                // Récupérer la colonne
                $whiteboard=$this->em->getRepository("App:Whiteboard")->find($id);

                // On recherche le derniers roworder pour ce whiteboard
                $last=$this->em->getRepository("App:Whiteboardwidget")->findOneBy(['whiteboard'=>$whiteboard],['roworder'=>'DESC']);
                $roworder=($last?$last->getRoworder()+1:1);

                // Récupérer le créateur
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                
                // On gérère le nouveau widget
                $wid= new Whiteboardwidget();
                $wid->setTitle("BILLET");
                $wid->setWhiteboard($whiteboard);
                $wid->setPosx(0);
                $wid->setPosy(0);
                $wid->setWidth(350);
                $wid->setRoworder($roworder);
                $wid->setUser($user);

                $this->em->persist($wid);
                $this->em->flush();
    
                // Send event
                $data->idwid = $wid->getId();
                $data->title = $wid->getTitle();
                $data->description = "";
                $data->username = ($user?$user->getUsername():"");
                $data->filetype = $wid->getFiletype();
                $data->file = $wid->getFile();
                $data->posx = $wid->getPosx();
                $data->posy = $wid->getPosy();
                $data->width = $wid->getWidth();
                $data->roworder = $wid->getRoworder();

                // Modification de l'url youtube
                if($data->filetype=="youtube"&&stripos($data->file,"https://www.youtube.com/embed")===false&&stripos($data->file,"youtube")!==false) {
                    $data->file=str_replace("http://www.youtube.com","https://www.youtube.com",$data->file);
                    $data->file=str_replace("https://www.youtube.com/watch?v=","",$data->file);    
                    $tmp=explode("&",$data->file);
                    $data->file="https://www.youtube.com/embed/".$tmp[0];                                         
                }

                // Modification de l'url peertube
                if($data->filetype=="youtube"&&stripos($data->file,"/watch/")!==false) {
                    $data->file=str_replace("/watch/","/embed/",$data->file);    
                }

                $this->sendMessage($conn,$data);
            break;   
            
            case "delwid" :
                // On recherche le widget pour le supprimer
                $wid=$this->em->getRepository("App:Whiteboardwidget")->find($data->id);
                if($wid) {
                    $this->em->remove($wid);
                    $this->em->flush(); 
                }                         
    
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "starteditwid" :
                // Send event
                $data->message= "modifie le billet";
                $this->sendMessage($conn,$data);
            break;

            case "editwid" :
                // On modifie le widget en cours
                $wid=$this->em->getRepository("App:Whiteboardwidget")->find($data->id);
                if($wid) {
                    // Modification de l'url youtube
                    if($data->filetype=="youtube"&&stripos($data->file,"https://www.youtube.com/embed")===false&&stripos($data->file,"youtube")!==false) {
                        $data->file=str_replace("http://www.youtube.com","https://www.youtube.com",$data->file);
                        $data->file=str_replace("https://www.youtube.com/watch?v=","",$data->file);     
                        $tmp=explode("&",$data->file);
                        $data->file="https://www.youtube.com/embed/".$tmp[0];                                       
                    }
                    
                    // Modification de l'url peertube
                    if($data->filetype=="youtube"&&stripos($data->file,"/watch/")!==false) {
                        $data->file=str_replace("/watch/","/embed/",$data->file);    
                    }

                    $wid->setTitle($data->title);
                    $wid->setDescription($data->description);
                    $wid->setFiletype($data->filetype);
                    $wid->setFile($data->file);
                    $wid->setColorbody($data->colorbody);
                    $wid->setColorfont($data->colorfont);

                    $this->em->persist($wid);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;
    
            case "canceleditwid" :
                // Send event
                $this->sendMessage($conn,$data);
            break;

            case "sizewid" :
                // Send event
                $this->sendMessage($conn,$data,"other");   
            break; 

            case "stopsizewid" :
                $wid=$this->em->getRepository("App:Whiteboardwidget")->find($data->id);
                if($wid) {
                    $wid->setWidth($data->width);

                    $this->em->persist($wid);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;

            case "movewid" :
                // Send event
                $this->sendMessage($conn,$data,"other");   
            break; 

            case "stopmovewid" :
                $wid=$this->em->getRepository("App:Whiteboardwidget")->find($data->id);
                if($wid) {
                    // On réordonne les widget car celui ci est forcement le dernier
                    $roworder=1;
                    $whiteboard=$this->em->getRepository("App:Whiteboard")->find($id);
                    $orders=$this->em->getRepository("App:Whiteboardwidget")->findBy(["whiteboard"=>$whiteboard],["roworder"=>"ASC"]);
                    $tborder=[];
                    foreach($orders as $order) {
                        if($order!=$wid) {
                            $order->setRoworder($roworder);
                            $this->em->persist($order);
                            $this->em->flush();                             
                            $tborder[$order->getId()]=$roworder;
                            $roworder=$roworder+1;
                        }
                    }
                    
                    $tborder[$data->id]=$roworder;
                    $wid->setPosY($data->posY);
                    $wid->setPosX($data->posX);
                    $wid->setRoworder($roworder);
                    
                    $data->tborder=$tborder;
                    $this->em->persist($wid);
                    $this->em->flush();                
                }
    
                // Send event
                $this->sendMessage($conn,$data);
            break;
                        
            case "startuplowid" :
                // Send event
                $data->message= "télécharge des PJ";
                $this->sendMessage($conn,$data);
            break;  
            
            case "stopuplowid" :
                // Send event
                $this->sendMessage($conn,$data);
            break;   
            
            case "addmessage" :
                $whiteboard=$this->em->getRepository("App:Whiteboard")->find($id);
                $wid=$this->em->getRepository("App:Whiteboardwidget")->find($data->id);
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                if($wid&&$user) {
                    // On gérère le nouveau message
                    $mes= new Message();
                    $mes->setMessage($data->message);
                    $mes->setWhiteboardwidget($wid);
                    $mes->setUser($user);

                    $this->em->persist($mes);
                    $this->em->flush();      
                    
                    $data->idmsg=$mes->getId();
                    $data->date=$mes->getSubmitdate()->format("d/m/Y H:i");
                    $data->usernamne=$mes->getUser()->getUsername();
                    $data->fgcandel=false;
                    $data->usermsg=$mes->getUser()->getId();
                    $data->userwid=$wid->getUser()->getId();
                    $data->userboa=$whiteboard->getUser()->getId();

                }
                $this->sendMessage($conn,$data);
            break;

            case "delmessage" :
                $msg=$this->em->getRepository("App:Message")->find($data->id);
                if($msg) {
                    $data->idwid=$msg->getWhiteboardwidget()->getId();
                    $this->em->remove($msg);
                    $this->em->flush(); 
                }

                $this->sendMessage($conn,$data);
            break;
        }
    }

    private function onMessageWordcloud(ConnectionInterface $conn, $data) {
        $wordCloudId=$this->channelkeys[$conn->resourceId];
        $this->debug("== onMessageWordcloud = $wordCloudId");
        $this->debug("== data->command = $data->command");

        switch ($data->command) {

            case "unnotify":
                $this->sendMessage($conn,$data);
            break;

            case "editword" :
                $this->debug("== editword = ".json_encode($data));

                // Récupérer le wordcloud
                $this->debug("- Get wordcloud ");
                $wordcloud=$this->em->getRepository("App:Wordcloud")->find($wordCloudId);
               
                // Récupérer le créateur
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                $perm = $data->perm;
                $this->debug("== perm = ".$perm);

                // Récupérer le word
                // les espaces sont transformés en "_" à l'affichage dans le nuage 
                // => Transformation de tous les mots, TOUS LES ESPACES SERONT DES "_" 
                $cleanWord = str_replace(' ', '_', $data->word);
                if($data->fgsubmit==0) {
                    // On change le mot de l'ancien mot
                    // si perm=write = on change le label de l'ensemble de l'instance de ce mot
                    // si perm=writeuser = on change le label de l'instance de ce mot et qui appartient à l'utilisateur

                    if($perm === 'write') {
                        $words = $this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$data->previousWord, 'wordcloud'=>$wordCloudId]);
                    } else {
                        $words = $this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$data->previousWord, 'user'=>$user]);  
                    }
                    foreach($words as $word) {
                        $word->setTitle($cleanWord);
                        $this->em->persist($word);
                    }
                    
                    // On change la couleur du mot
                    // si perm=write = on change le label de l'ensemble de l'instance de ce mot
                    // si perm=writeuser = on change le label de l'instance de ce mot et qui appartient à l'utilisateur
                    if($perm === 'write') {
                        $words = $this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$cleanWord, 'wordcloud'=>$wordCloudId]);
                    } else {
                        $words = $this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$cleanWord, 'user'=>$user]);  
                    }
                    foreach($words as $word) {
                        $word->setColorfont($data->color);
                        $this->em->persist($word);
                    }
                }
                else {
                    $words = $this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$cleanWord, 'wordcloud'=>$wordCloudId]);
                    $wordColor = BLACK_COLOR;
                    if($words){
                        foreach($words as $word) {
                            $wordColor = ($word->getColorfont() != BLACK_COLOR) ? $word->getColorfont() : $data->color;
                        }
                    }
                    $this->debug("- submit");
                    $word= new Wordcloudword();

                    // On gérère le nouveau widget
                    $word->setTitle($cleanWord);
                    $word->setColorfont($wordColor);
                    $word->setUser($user);
                    $word->setWordcloud($wordcloud);
    
                    $this->em->persist($word);
                }
                $this->em->flush();
                $this->debug("== flush = ");

                $this->sendMessage($conn,$data);
            break;   
            
            case "delword" :
                $wordcloud=$this->em->getRepository("App:Wordcloud")->find($wordCloudId);
                $user=$this->em->getRepository("App:User")->findOneBy(["username"=>$data->username]);
                $perm = $this->getWordCloudPermission($wordcloud, $user);

                $this->debug("== delword");
                $this->debug("== delword perm".$perm);
                // TODO A VERIFIER SI OK
                // todo write => suppression de toutes les intances du mots
                // todo writeuser => suppression de son instance
                if($perm === 'write') {
                    $words=$this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$data->word]);
                } else {
                    $words=$this->em->getRepository("App:Wordcloudword")->findBy(['title'=>$data->word, 'user'=>$user]);
                }
                 // On recherche le widget pour le supprimer

                if($words) {
                    foreach($words as $word) {
                        $this->em->remove($word);
                    }
                    $this->em->flush(); 
                }
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "starteditword" :
                // Send event
                $data->message= "création/modification de mot";
                $this->sendMessage($conn,$data);
            break;

            case "canceleditword" :
                // Send event
                $this->sendMessage($conn,$data);
            break;
        }
    }

    private function onMessageTimeline(ConnectionInterface $conn, $data) {
        $id=$this->channelkeys[$conn->resourceId];

        switch ($data->command) {

            case "unnotify":
                $this->sendMessage($conn,$data);
            break;

            case "startaddsegment":
                // Send event
                $data->message= "ajoute un segment";
                $this->sendMessage($conn,$data);
            break; 

            case "annaddsegment" :
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "valaddsegment" :
                // Récupérer la Timeline
                $timeline=$this->em->getRepository("App:Timeline")->find($id);

                if($data->startmonth=="") $data->startmonth=0;
                if($data->endmonth=="") $data->endmonth=0;

                // On ajoute un segment
                $seg= new Timelinesegment();
                $seg->setTitle($data->title);
                $seg->setDescription($data->description);
                $seg->setStart($data->start);
                $seg->setStartmonth($data->startmonth);
                $seg->setEnd($data->end);
                $seg->setEndmonth($data->endmonth);
                $seg->setColor($data->color);
                $seg->setUsebackground($data->usebackground);
                $seg->setBackground($data->background);
                $seg->setLine($data->line);
                $seg->setTimeline($timeline);
                $this->em->persist($seg);
                $this->em->flush();

                // Retour
                $data->segment= new \stdClass;
                $data->segment->id = $seg->getId();
                $data->segment->title = $seg->getTitle();
                $data->segment->start = $seg->getStart();
                $data->segment->startmonth = $seg->getStartmonth();
                $data->segment->end = $seg->getEnd();
                $data->segment->endmonth = $seg->getEndmonth();
                $data->segment->color = ($seg->getColor()?$seg->getColor():$this->getConfig("colorbgbodydark"));
                $data->segment->usebackground = $seg->getUsebackground();
                $data->segment->background = $seg->getBackground();
                $data->segment->line = $seg->getLine();
                $this->sendMessage($conn,$data);                
            break;

            case "startmodsegment":
                // Send event
                $data->message= "modifie un segment";
                $this->sendMessage($conn,$data);
            break; 

            case "annmodsegment" :
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "valmodsegment" :
                // Récupérer le Timelinesegment
                $seg=$this->em->getRepository("App:Timelinesegment")->find($data->id);

                if($seg) {
                    if($data->startmonth=="") $data->startmonth=0;
                    if($data->endmonth=="") $data->endmonth=0;

                    // On ajoute un segment
                    $seg->setTitle($data->title);
                    $seg->setDescription($data->description);
                    $seg->setStart($data->start);
                    $seg->setStartmonth($data->startmonth);
                    $seg->setEnd($data->end);
                    $seg->setEndmonth($data->endmonth);
                    $seg->setColor($data->color);
                    $seg->setUsebackground($data->usebackground);
                    $seg->setBackground($data->background);
                    $seg->setLine($data->line);
                    $this->em->persist($seg);
                    $this->em->flush();

                    // Retour
                    $data->segment= new \stdClass;
                    $data->segment->id = $seg->getId();
                    $data->segment->title = $seg->getTitle();
                    $data->segment->start = $seg->getStart();
                    $data->segment->startmonth = $seg->getStartmonth();
                    $data->segment->end = $seg->getEnd();
                    $data->segment->endmonth = $seg->getEndmonth();
                    $data->segment->color = ($seg->getColor()?$seg->getColor():$this->getConfig("colorbgbodydark"));
                    $data->segment->usebackground = $seg->getUsebackground();
                    $data->segment->background = $seg->getBackground();
                    $data->segment->line = $seg->getLine();
                }

                $this->sendMessage($conn,$data);                
            break;

            case "delsegment" :
                // Récupérer le Timelinesegment
                $seg=$this->em->getRepository("App:Timelinesegment")->find($data->id);
                if($seg) {
                    $this->em->remove($seg);
                    $this->em->flush(); 
                }                         
    
                // Send event
                $this->sendMessage($conn,$data);   
            break;
            
            case "startadddate":
                // Send event
                $data->message= "ajoute une date";
                $this->sendMessage($conn,$data);
            break; 

            case "annadddate" :
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "valadddate" :
                // Récupérer la Timeline
                $timeline=$this->em->getRepository("App:Timeline")->find($id);

                if($data->startmonth=="") $data->startmonth=0;
                if($data->startday=="") $data->startday=0;
                                
                // On ajoute un date
                $date= new Timelinedate();
                $date->setTitle($data->title);
                $date->setDescription($data->description);
                $date->setStart($data->start);
                $date->setStartmonth($data->startmonth);
                $date->setStartday($data->startday);
                $date->setColor($data->color);
                $date->setUsebackground($data->usebackground);
                $date->setBackground($data->background);
                $date->setPosy(0);
                $date->setTimeline($timeline);
                $this->em->persist($date);
                $this->em->flush();

                // Retour
                $data->date= new \stdClass;
                $data->date->id = $date->getId();
                $data->date->title = $date->getTitle();
                $data->date->start = $date->getStart();
                $data->date->startmonth = $date->getStartmonth();
                $data->date->startday = $date->getStartday();
                $data->date->color = ($date->getColor()?$seg->getColor():$this->getConfig("colorbgbodydark"));
                $data->date->usebackground = $date->getUsebackground();
                $data->date->background = $date->getBackground();
                $data->date->posY = $date->getPosy();
                $this->sendMessage($conn,$data);                
            break;   
            
            case "startmoddate":
                // Send event
                $data->message= "modifie une date";
                $this->sendMessage($conn,$data);
            break; 

            case "annmoddate" :
                // Send event
                $this->sendMessage($conn,$data);
            break;    
            
            case "valmoddate" :
                // Récupérer le Timelinedate
                $date=$this->em->getRepository("App:Timelinedate")->find($data->id);

                if($date) {
                    if($data->startmonth=="") $data->startmonth=0;
                    if($data->startday=="") $data->startday=0;

                    // On modifie une date
                    $date->setTitle($data->title);
                    $date->setDescription($data->description);
                    $date->setStart($data->start);
                    $date->setStartmonth($data->startmonth);
                    $date->setStartday($data->startday);
                    $date->setColor($data->color);
                    $date->setUsebackground($data->usebackground);
                    $date->setBackground($data->background);
                    $this->em->persist($date);
                    $this->em->flush();

                    // Retour
                    $data->date= new \stdClass;
                    $data->date->id = $date->getId();
                    $data->date->title = $date->getTitle();
                    $data->date->start = $date->getStart();
                    $data->date->startmonth = $date->getStartmonth();
                    $data->date->startday = $date->getStartday();
                    $data->date->color = ($date->getColor()?$date->getColor():$this->getConfig("colorbgbodydark"));
                    $data->date->usebackground = $date->getUsebackground();
                    $data->date->background = $date->getBackground();
                    $data->date->posY = $date->getPosy();
                }

                $this->sendMessage($conn,$data);                
            break;

            case "deldate" :
                // Récupérer le Timelinedate
                $date=$this->em->getRepository("App:Timelinedate")->find($data->id);
                if($date) {
                    $this->em->remove($date);
                    $this->em->flush(); 
                }                         
    
                // Send event
                $this->sendMessage($conn,$data);   
            break; 
            
            case "movedate" :
                // Send event
                $this->sendMessage($conn,$data,"other");   
            break; 

            case "stopmovedate" :
                // Récupérer le Timelinedate
                $date=$this->em->getRepository("App:Timelinedate")->find($data->id);

                if($date) {
                    $date->setPosY($data->posY);
                    $this->em->persist($date);
                    $this->em->flush();

                    // Pas de message à envoyer
                }
            break;            
        }
    }

    private function onMessageMindmap(ConnectionInterface $conn, $data) {
        $id=$this->channelkeys[$conn->resourceId];

        switch ($data->command) {

            case "unnotify":
                $this->sendMessage($conn,$data);
            break;

            case "update":
                // Récupérer la Mindmap
                $mindmap=$this->em->getRepository("App:Mindmap")->find($id);
                $mindmap->setDescription($data->description);

                $this->em->persist($mindmap);
                $this->em->flush();

                $this->sendMessage($conn,$data,"other");                
            break;

            
            case "starteditwid" :
                // Send event
                $data->message= "modifie l'idée ".$data->id. " = ".$data->label;
                $this->sendMessage($conn,$data);
            break;            
        }
    }

    private function sendMessage(ConnectionInterface $conn, $data, $dest="all") {
        $target = $this->channels[$conn->resourceId];
        foreach ($this->channels as $id=>$channel) {
            if ($channel == $target) {
                // dest = all - tout le monde meme l'expediteur
                // dest = other - tout le monde sauf l'expedituer
                // dest = me - seumlement l'expediteur
                if($dest=="all"||($dest=="other"&&$id!=$conn->resourceId)||($dest=="me"&&$id==$conn->resourceId)) {
                    // From
                    $key= $this->userkeys[$conn->resourceId];
                    $from=$this->em->getRepository("App:User")->findOneBy(["apikey"=>$key]);

                    // To
                    $key= $this->userkeys[$id];
                    $to=$this->em->getRepository("App:User")->findOneBy(["apikey"=>$key]);

                    // Send
                    if($from && $to) {
                        $data->from= new \stdClass;
                        $data->from->id             = $from->getId();
                        $data->from->username       = $from->getUsername();
                        $data->from->displayname    = $from->getDisplayname();
                        $data->from->avatar         = $this->getAvatar($from->getAvatar());
                        
                        $data->log="== GET MSG from ".$data->from->username." to ".$to->getUsername()." = ".$data->command;
                        $this->userconns[$id]->send(json_encode($data));
                    }
                }
            }
        }
    }

    private function getAvatar($avatar) {
        if(stripos($avatar,"http")===0)
            return $avatar;
        else
            return "/".$this->container->getParameter("appAlias")."/uploads/avatar/".$avatar;        
    }

    private function getConfig($id) {
        $val="";
        $config=$this->em->getRepository("App:Config")->findOneBy(['id'=>$id]);
        if($config) {
            $val=$config->getValue();
            if($val=="")
                $val = $config->getDefault();
        }

        return $val;
    }

    private function getWordCloudPermission($entity, User $currentUser): string {
        $perm=false;

        if($entity->getUser()==$currentUser) $perm="write";
        if($currentUser->hasRole("ROLE_ADMIN")) $perm="write";
        if($currentUser->hasRole("ROLE_MODO")) $perm="write";

        // Récupération des groupes de l'utilisateur
        $groups=$currentUser->getGroups();

        // Ses groupes ont-ils la permission 
        foreach($groups as $group) {
            if($entity->getGroups()->contains($group)) $perm="write";
        }

        // Son compte a-til la permission
        if($entity->getUsers()->contains($currentUser)) $perm="write";

        // Si pas de permission on regarde s'il a les permissions en écriture limitée
        if(!$perm) {
            // Ses groupes ont-ils la permission 
            foreach($groups as $group) {
                if($entity->getGroupwriters()->contains($group)) $perm="writeuser";
            }

            // Son compte a-til la permission
            if($entity->getUserwriters()->contains($currentUser)) $perm="writeuser";
        }

        // Si pas de permission on regarde s'il a les permissions en lecture
        if(!$perm) {
            // Ses groupes ont-ils la permission 
            foreach($groups as $group) {
                if($entity->getGroupreaders()->contains($group)) $perm="read";
            }

            // Son compte a-til la permission
            if($entity->getUserreaders()->contains($currentUser)) $perm="read";
        }

        // Erreur si non permis
        if(!$perm) $perm="unauthorized";

        return $perm;
    }
}