(****************************************************************************)
(*     Sail                                                                 *)
(*                                                                          *)
(*  Sail and the Sail architecture models here, comprising all files and    *)
(*  directories except the ASL-derived Sail code in the aarch64 directory,  *)
(*  are subject to the BSD two-clause licence below.                        *)
(*                                                                          *)
(*  The ASL derived parts of the ARMv8.3 specification in                   *)
(*  aarch64/no_vector and aarch64/full are copyright ARM Ltd.               *)
(*                                                                          *)
(*  Copyright (c) 2013-2021                                                 *)
(*    Kathyrn Gray                                                          *)
(*    Shaked Flur                                                           *)
(*    Stephen Kell                                                          *)
(*    Gabriel Kerneis                                                       *)
(*    Robert Norton-Wright                                                  *)
(*    Christopher Pulte                                                     *)
(*    Peter Sewell                                                          *)
(*    Alasdair Armstrong                                                    *)
(*    Brian Campbell                                                        *)
(*    Thomas Bauereiss                                                      *)
(*    Anthony Fox                                                           *)
(*    Jon French                                                            *)
(*    Dominic Mulligan                                                      *)
(*    Stephen Kell                                                          *)
(*    Mark Wassell                                                          *)
(*    Alastair Reid (Arm Ltd)                                               *)
(*                                                                          *)
(*  All rights reserved.                                                    *)
(*                                                                          *)
(*  This work was partially supported by EPSRC grant EP/K008528/1 <a        *)
(*  href="http://www.cl.cam.ac.uk/users/pes20/rems">REMS: Rigorous          *)
(*  Engineering for Mainstream Systems</a>, an ARM iCASE award, EPSRC IAA   *)
(*  KTF funding, and donations from Arm.  This project has received         *)
(*  funding from the European Research Council (ERC) under the European     *)
(*  Union’s Horizon 2020 research and innovation programme (grant           *)
(*  agreement No 789108, ELVER).                                            *)
(*                                                                          *)
(*  This software was developed by SRI International and the University of  *)
(*  Cambridge Computer Laboratory (Department of Computer Science and       *)
(*  Technology) under DARPA/AFRL contracts FA8650-18-C-7809 ("CIFV")        *)
(*  and FA8750-10-C-0237 ("CTSRD").                                         *)
(*                                                                          *)
(*  SPDX-License-Identifier: BSD-2-Clause                                   *)
(****************************************************************************)

(** This file implements utilities for dealing with $property and $counterexample pragmas. *)

open Ast
open Ast_defs
open Ast_util
open Type_check

(** [find_properties defs] returns a mapping from ids to of 4-tuples of the form (prop_type, command, loc, val_spec),
    which contains the information from any pragmas of the form

    $prop_type command ... val <val_spec>

    where prop_type is either "counterexample" or "property" and the location loc is the location that was attached to
    the pragma *)
val find_properties : ('a, 'b) ast -> (string * string * l * 'a val_spec) Bindings.t

(** For a property

    $prop_type val f : forall X, C. T -> bool

    find the function body for id:

    function f(args) = exp

    and rewrite the function body to

    function f(args) = if constraint(not(C)) then true else exp

    The reason we do this is that the type information in T constrained by C might be lost when translating to Jib, as
    Jib types are simpler and less precise. If we then do random test generation/proving we want to ensure that inputs
    outside the constraints of the function are ignored. *)
val rewrite : typed_ast -> typed_ast

type event = Overflow | Assertion | Assumption | Match | Return

val string_of_event : event -> string

module Event : sig
  type t = event
  val compare : event -> event -> int
end

type query = Q_all of event | Q_exist of event | Q_not of query | Q_and of query list | Q_or of query list

val default_query : query

type pragma = { query : query; litmus : string list }

val parse_pragma : Parse_ast.l -> string -> pragma
