% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxpath.R
\name{cox.path}
\alias{cox.path}
\title{Fit a Cox regression model with elastic net regularization for a path of
lambda values}
\usage{
cox.path(
  x,
  y,
  weights = NULL,
  offset = NULL,
  alpha = 1,
  nlambda = 100,
  lambda.min.ratio = ifelse(nobs < nvars, 0.01, 1e-04),
  lambda = NULL,
  standardize = TRUE,
  thresh = 1e-10,
  exclude = NULL,
  penalty.factor = rep(1, nvars),
  lower.limits = -Inf,
  upper.limits = Inf,
  maxit = 1e+05,
  trace.it = 0,
  ...
)
}
\arguments{
\item{x}{See glmnet help file}

\item{y}{Survival response variable, must be a \code{Surv} or
\code{stratifySurv} object.}

\item{weights}{See glmnet help file}

\item{offset}{See glmnet help file}

\item{alpha}{See glmnet help file}

\item{nlambda}{See glmnet help file}

\item{lambda.min.ratio}{See glmnet help file}

\item{lambda}{See glmnet help file}

\item{standardize}{See glmnet help file}

\item{thresh}{Convergence threshold for coordinate descent. Each inner
coordinate-descent loop continues until the maximum change in the objective
after any coefficient update is less than thresh times the null deviance.
Default value is \code{1e-10}.}

\item{exclude}{See glmnet help file}

\item{penalty.factor}{See glmnet help file}

\item{lower.limits}{See glmnet help file}

\item{upper.limits}{See glmnet help file}

\item{maxit}{See glmnet help file}

\item{trace.it}{Controls how much information is printed to screen. Default is
\code{trace.it=0} (no information printed). If \code{trace.it=1}, a progress
bar is displayed. If \code{trace.it=2}, some information about the fitting
procedure is printed to the console as the model is being fitted.}

\item{...}{Other arguments passed from glmnet (not used right now).}
}
\value{
An object of class "coxnet" and "glmnet".
\item{a0}{Intercept value, \code{NULL} for "cox" family.}
\item{beta}{A \code{nvars x length(lambda)} matrix of coefficients, stored in
sparse matrix format.}
\item{df}{The number of nonzero coefficients for each value of lambda.}
\item{dim}{Dimension of coefficient matrix.}
\item{lambda}{The actual sequence of lambda values used. When alpha=0, the
largest lambda reported does not quite give the zero coefficients reported
(lambda=inf would in principle). Instead, the largest lambda for alpha=0.001
is used, and the sequence of lambda values is derived from this.}
\item{dev.ratio}{The fraction of (null) deviance explained. The deviance
calculations incorporate weights if present in the model. The deviance is
defined to be 2*(loglike_sat - loglike), where loglike_sat is the log-likelihood
for the saturated model (a model with a free parameter per observation).
Hence dev.ratio=1-dev/nulldev.}
\item{nulldev}{Null deviance (per observation). This is defined to be
2*(loglike_sat -loglike(Null)). The null model refers to the 0 model.}
\item{npasses}{Total passes over the data summed over all lambda values.}
\item{jerr}{Error flag, for warnings and errors (largely for internal
debugging).}
\item{offset}{A logical variable indicating whether an offset was included
in the model.}
\item{call}{The call that produced this object.}
\item{nobs}{Number of observations.}
}
\description{
Fit a Cox regression model via penalized maximum likelihood for a path of
lambda values. Can deal with (start, stop] data and strata, as well as
sparse design matrices.
}
\details{
Sometimes the sequence is truncated before \code{nlambda} values of lambda
have been used. This happens when \code{cox.path} detects that the
decrease in deviance is marginal (i.e. we are near a saturated fit).
}
\examples{
set.seed(2)
nobs <- 100; nvars <- 15
xvec <- rnorm(nobs * nvars)
xvec[sample.int(nobs * nvars, size = 0.4 * nobs * nvars)] <- 0
x <- matrix(xvec, nrow = nobs)
beta <- rnorm(nvars / 3)
fx <- x[, seq(nvars / 3)] \%*\% beta / 3
ty <- rexp(nobs, exp(fx))
tcens <- rbinom(n = nobs, prob = 0.3, size = 1)
jsurv <- survival::Surv(ty, tcens)
fit1 <- glmnet:::cox.path(x, jsurv)

# works with sparse x matrix
x_sparse <- Matrix::Matrix(x, sparse = TRUE)
fit2 <- glmnet:::cox.path(x_sparse, jsurv)

# example with (start, stop] data
set.seed(2)
start_time <- runif(100, min = 0, max = 5)
stop_time <- start_time + runif(100, min = 0.1, max = 3)
status <- rbinom(n = nobs, prob = 0.3, size = 1)
jsurv_ss <- survival::Surv(start_time, stop_time, status)
fit3 <- glmnet:::cox.path(x, jsurv_ss)

# example with strata
jsurv_ss2 <- stratifySurv(jsurv_ss, rep(1:2, each = 50))
fit4 <- glmnet:::cox.path(x, jsurv_ss2)
}
