use std::sync::Arc;

use axum::{
    response::{Html, IntoResponse},
    routing::get,
    Extension, Router,
};
use indieweb::standards::indieauth::{self, ClientId};
use miette::IntoDiagnostic;
use minijinja::{Environment, Value};
use tower_livereload::LiveReloadLayer;

fn engine() -> minijinja::Environment<'static> {
    let mut env = Environment::new();
    minijinja_embed::load_templates!(&mut env);
    env
}

mod client;

#[tracing::instrument]
async fn landing() -> impl IntoResponse {
    Html(
        engine()
            .get_template("index.html")
            .unwrap()
            .render(Value::default())
            .unwrap(),
    )
}

#[derive(Clone)]
pub struct SharedState {
    client_id: ClientId,
    client: Arc<indieauth::Client<indieweb::http::reqwest::Client>>,
}

impl SharedState {
    fn new() -> miette::Result<Self> {
        let client_id = ClientId::new("http://127.0.0.1:18000")?;
        let client = Arc::new(indieauth::Client::new(
            client_id.as_str(),
            indieweb::http::reqwest::Client::default(),
        )?);
        Ok(Self { client_id, client })
    }
}

#[tokio::main]
#[tracing::instrument]
async fn main() -> miette::Result<()> {
    tracing_forest::init();
    let livereload_layer = LiveReloadLayer::new();

    let app: Router<()> = Router::new()
        .route("/", get(landing))
        .nest("/client", client::router())
        .layer(Extension(SharedState::new()?))
        .layer(livereload_layer)
        .with_state(());

    let listener = tokio::net::TcpListener::bind("127.0.0.1:18000")
        .await
        .into_diagnostic()?;

    tracing::trace!("Listening at http://127.0.0.1:18000 with a sample IndieWeb client server");
    axum::serve(listener, app).await.into_diagnostic()?;

    Ok(())
}
