#-*-coding:utf-8-*-
"""
    Tests de la gestion des partages associés à un groupe
"""
from os.path import isdir, join
from os import listdir
from time import sleep
from subprocess import getstatusoutput
from scribe.ldapconf import CLASS_PATH, GROUP_PATH, CONTAINER_PATH_FICHIER
from scribe.eolegroup import Group
from scribe.eoleshare import Share
from py.test import raises
import time

def getfacl(group_name, dirname, default=False):
    grep = 'group:%s:' % group_name
    if default:
        grep = 'default:%s' % grep
    ret_code, ret_value = getstatusoutput("getfacl %s 2>/dev/null | grep ^%s " % (dirname, grep))
    if not ret_code % 256:
        return ret_value
    else:
        raise Exception("Erreur dans getfacl")


class TestShare:
    """
        test la création des partages de type 'simples'
        en lecture-écriture ou lecture seule
    """
    sh_name = 'test__eoleshare__sharerw__'
    sh_name_ro = 'test__eoleshare__sharero__'
    gr_name = 'test__eoleshare__group__'
    dirname = join(GROUP_PATH, 'test__eoleshare__sharerw__')
    dirname_ro = join(GROUP_PATH, 'test__eoleshare__sharero__')

    def setup_class(cls):
        cls.sh = Share()
        cls.gr = Group()
        cls.sh.ldap_admin.connect()

    def test_add(self):
        self.gr.add_groupe(self.gr_name)
        self.sh._add(self.sh_name, self.gr_name, 'rw', sync=False)
        self.sh._add(self.sh_name_ro, self.gr_name, 'ro', filepath=self.dirname_ro, sync=False)
        self.sh._synchronize(False)

    def test_ldap(self):
        assert self.sh._get_attr(self.sh_name, 'sambaShareGroup') == [self.gr_name]
        assert self.sh._get_attr(self.sh_name, 'sambaFilePath') == [self.dirname]

    def test_dir(self):
        assert isdir(self.dirname)
        assert getfacl(self.gr_name, self.dirname) == 'group:%s:rwx' % self.gr_name
        assert getfacl(self.gr_name, self.dirname, True) == 'default:group:%s:rwx' % self.gr_name
        assert getfacl('', self.dirname) == 'group::---'
        assert getfacl('', self.dirname, True) == 'default:group::---'

    def test_ro_dir(self):
        assert getfacl(self.gr_name, self.dirname_ro) == 'group:%s:r-x' % self.gr_name
        assert getfacl(self.gr_name, self.dirname_ro, True) == 'default:group:%s:r-x' % self.gr_name

    def test_synchronize(self):
        smb_conf = open(join(CONTAINER_PATH_FICHIER, 'etc/samba/conf.d/eole.conf'), 'r').read()
        assert "[%s]" % self.sh_name in smb_conf and "[%s]" % self.sh_name_ro in smb_conf

    def test_drive(self):
        if self.sh._get_share('Z:') == "":
            self.sh._set_share_drive(self.sh_name, 'z')
            assert self.sh._get_attr(self.sh_name, 'sambaShareDrive') == ['Z:']
        else:
            raises(Exception, self.sh._set_share_drive, self.sh_name, 'z')
        # partage réservé Eole
        raises(Exception, self.sh._set_share_drive, 'professeurs', 'P:')
        raises(Exception, self.sh._set_share_drive, 'self.sh_name', 'azerty')

    def test_delete(self):
        self.sh._delete(self.sh_name, rmdir=True, sync=False)
        self.sh._delete(self.sh_name_ro, rmdir=True, sync=False)
        assert not isdir(self.dirname)
        time.sleep(10)
        assert self.sh._get_attr(self.sh_name, 'sambaShareGroup') == []
        self.gr.delete(self.gr_name)
        self.sh._synchronize(False)

    def test_desynchronize(self):
        smb_conf = open(join(CONTAINER_PATH_FICHIER, 'etc/samba/smb.conf'), 'r').read()
        assert not("[%s]" % self.sh_name in smb_conf and "[%s]" % self.sh_name_ro in smb_conf)

class TestDTShare:
    """
        test la création des partages de type données/travail
    """
    sh_name = 'test__eoleshare__sharedt__'
    gr_name = 'test__eoleshare__groupdt__'
    dirname = join(GROUP_PATH, 'test__eoleshare__sharedt__')
    datadirname = join(dirname, 'donnees')
    workdirname = join(dirname, 'travail')

    def setup_class(cls):
        cls.gr = Group()
        cls.sh = Share()

    def test_add(self):
        self.gr.add_groupe(self.gr_name)
        self.sh.add(self.sh_name, self.gr_name, 'dt', sync=False)

    def test_ldap(self):
        assert self.sh.get_attr(self.sh_name, 'sambaShareGroup') == [self.gr_name]
        assert self.sh.get_attr(self.sh_name, 'sambaFilePath') == [self.dirname]

    def test_dir(self):
        assert isdir(self.dirname)
        assert getfacl("professeurs", self.dirname) == 'group:professeurs:r-x'
        # acls du groupe
        assert getfacl(self.gr_name, self.dirname) == 'group:%s:r-x' % self.gr_name
        assert getfacl(self.gr_name, self.dirname, True) == 'default:group:%s:r-x' % self.gr_name
        # acl du groupe propriétaire
        assert getfacl('', self.dirname) == 'group::---'
        assert getfacl('', self.dirname, True) == 'default:group::---'

    def test_data_dir(self):
        assert isdir(self.datadirname)
        assert getfacl(self.gr_name, self.datadirname) == 'group:%s:r-x' % self.gr_name
        assert getfacl("professeurs", self.datadirname) == 'group:professeurs:rwx'

    def test_work_dir(self):
        assert isdir(self.workdirname)
        assert getfacl(self.gr_name, self.workdirname) == 'group:%s:rwx' % self.gr_name
        assert getfacl("professeurs", self.workdirname) == 'group:professeurs:rwx'

    def test_delete(self):
        self.sh.delete(self.sh_name, rmdir=True, sync=False)
        assert not isdir(self.dirname)
        sleep(0.5)
        assert self.sh.get_attr(self.sh_name, 'sambaShareGroup') == []
        self.gr.delete(self.gr_name)


