/*
  Copyright (C) 2024 UBports Foundation.

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License 3 as published by
  the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program. If not, see http://www.gnu.org/licenses/.
*/

#include <QGuiApplication>
#include <QLibrary>
#include <QQmlApplicationEngine>
#include <QQuickView>
#include <QStandardPaths>
#include <QQuickWindow>

#include <libintl.h>

#include "config.hpp"

int main (int argc, char *argv[])
{
    qputenv ("QT_PRINTER_MODULE", "cupsprintersupport");

    QGuiApplication app (argc, argv);
    app.setOrganizationName (QStringLiteral ("UBports"));
    app.setApplicationName (QStringLiteral ("lomiri-printing-app"));
    // app.setApplicationVersion (AppVersion());

    std::string domainName = localeDomain().toStdString();
    std::string domainDir = localeDirectory().toStdString();

    textdomain (domainName.c_str());
    bindtextdomain (domainName.c_str(), domainDir.c_str());
    bind_textdomain_codeset (domainName.c_str(), "UTF-8");

    QQuickView view;

    QObject::connect ((QObject*) view.engine(), SIGNAL (quit()), &app, SLOT (quit()));

    // Set up import paths
    QStringList importPathList = view.engine()->importPathList();
    importPathList.prepend (qmlDirectory());

    QStringList args = app.arguments();
    if (args.contains("-h") || args.contains("--help")) {
        qDebug() << "usage: " + args.at(0) + " [-h|--help] [-I <path>] [-q <qmlfile>]";
        qDebug() << "    -h|--help     Print this help.";
        qDebug() << "    -I <path>     Give a path for an additional QML import directory. May be used multiple times.";
        qDebug() << "    -q <qmlfile>  Give an alternative location for the main qml file.";
        return 0;
    }

    QString qmlfile;
    for (int i = 0; i < args.count(); i++) {
        if (args.at (i) == "-I" && args.count() > i + 1) {
            QString addedPath = args.at (i+1);
            if (addedPath.startsWith ('.')) {
                addedPath = addedPath.right (addedPath.length() - 1);
                addedPath.prepend (QDir::currentPath());
            }
            importPathList.append (addedPath);
        } else if (args.at (i) == "-q" && args.count() > i + 1) {
            qmlfile = args.at (i+1);
        }
    }

    if (args.contains (QLatin1String ("-testability")) || getenv ("QT_LOAD_TESTABILITY")) {
        QLibrary testLib (QLatin1String ("qttestability"));
        if (testLib.load()) {
            typedef void (*TasInitialize)(void);
            TasInitialize initFunction = (TasInitialize) testLib.resolve("qt_testability_init");
            if (initFunction) {
                initFunction();
            } else {
                qCritical("Library qttestability resolve failed!");
            }
        } else {
            qCritical("Library qttestability load failed!");
        }
    }

    view.engine()->setImportPathList (importPathList);

    // load the qml file
    if (qmlfile.isEmpty()) {
        QStringList paths = QStandardPaths::standardLocations (QStandardPaths::DataLocation);
        paths.prepend (QDir::currentPath());
        paths.prepend (QCoreApplication::applicationDirPath());
        paths.prepend (qmlDirectory());

        foreach (const QString &path, paths) {
            QFileInfo fi (path + "/Main.qml");
            qDebug() << "Trying to load QML from:" << fi.filePath();
            if (fi.exists()) {
                qmlfile = fi.filePath();
                break;
            }
        }
    }
    qDebug() << "using main qml file from:" << qmlfile;
    view.setSource (QUrl::fromLocalFile (qmlfile));
    view.show();
    return app.exec();

}
