package latexDraw.generators.svg;

import java.awt.Color;

import latexDraw.figures.ArrowHead;
import latexDraw.figures.BezierCurve;
import latexDraw.figures.Line;
import latexDraw.figures.properties.Arrowable;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.SVGDefsElement;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGPathElement;
import latexDraw.parsers.svg.elements.path.SVGPathSegClosePath;
import latexDraw.parsers.svg.elements.path.SVGPathSegCurvetoCubic;
import latexDraw.parsers.svg.elements.path.SVGPathSegList;
import latexDraw.parsers.svg.elements.path.SVGPathSegMoveto;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for a Bézier curve.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LBezierCurveSVGGenerator extends LPolygonSVGGenerator
{
	public LBezierCurveSVGGenerator(BezierCurve f)
	{
		super(f);
	}
	
	
	public LBezierCurveSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	/**
	 * Creates a Bézier curve from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LBezierCurveSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new BezierCurve(false));

		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPathElement))
			throw new IllegalArgumentException();
		
		SVGPathElement main = (SVGPathElement)elt2;
		BezierCurve bc = (BezierCurve)getShape();
		ArrowHead arrowHead1 = bc.getArrowHead1();
		ArrowHead arrowHead2 = bc.getArrowHead2();
		
		setPath(main.getSegList());
		setNumber(elt);
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		LaTeXDrawPoint2D pt1 = new LaTeXDrawPoint2D(bc.getPoint(0));
		LaTeXDrawPoint2D pt2 = new LaTeXDrawPoint2D(bc.getPoint(-1));
		
		arrowHead1.setPosition(pt1);
		arrowHead1.setLine(new Line(pt1, bc.getFirstControlPoint(0), false));
		arrowHead1.setFigure(bc);
		arrowHead2.setPosition(pt2);
		arrowHead2.setLine(new Line(pt2, bc.getFirstControlPoint(-1), false));
		arrowHead2.setFigure(bc);
		
		setSVGArrow(arrowHead1, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_START), main);
		setSVGArrow(arrowHead2, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_END), main);
		homogeniseArrows(arrowHead1, arrowHead2);
		
		bc.setShowPoints(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHOW_PTS)!=null);
		bc.updateShape();
		bc.updateBorders();
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	/**
	 * Sets the shape path according to the given SVG path segments.
	 * @param list The SVG path segments list
	 * @since 2.0.0
	 */
	public void setPath(SVGPathSegList list)
	{
		if(list==null || list.size()<2 || !(list.firstElement() instanceof SVGPathSegMoveto))
			throw new IllegalArgumentException();
		
		BezierCurve bc = (BezierCurve)getShape();
		SVGPathSegMoveto m = (SVGPathSegMoveto)list.firstElement();
		SVGPathSegCurvetoCubic c;
		int i=1, size = list.size();
		
		bc.addPoint(new LaTeXDrawPoint2D(m.getX(), m.getY()));
		
		if(i>0 && list.elementAt(1) instanceof SVGPathSegCurvetoCubic)
		{// We set the control point of the first point.
			c = (SVGPathSegCurvetoCubic)list.elementAt(1);
			bc.getFirstControlPoint(-1).setLocation(c.getX1(), c.getY1());
		}
		
		while(i<size && list.elementAt(i) instanceof SVGPathSegCurvetoCubic)
		{
			c = (SVGPathSegCurvetoCubic)list.elementAt(i);
			bc.addPoint(new LaTeXDrawPoint2D(c.getX(), c.getY()));
			bc.getFirstControlPoint(-1).setLocation(c.getX2(), c.getY2());
			i++;
		}
		
		bc.removePointAt(0); // We remove the two first points created during the first initialisation.
		bc.removePointAt(0);
		
		if(bc.getPoint(-1).equals(bc.getPoint(0), 0.00001))// We set the shape as closed
		{
			bc.removePointAt(-1);
			bc.setOpen(false);
			bc.setCloseType(BezierCurve.CLOSE_TYPE_CURVE);
		}
		else
			if(i<size && list.elementAt(i) instanceof SVGPathSegClosePath) // There is something else at the end of the path.
			{
				bc.setOpen(false);
				bc.setCloseType(BezierCurve.CLOSE_TYPE_LINE);
			}
			else
				bc.setOpen(true);
		
		bc.updateSecondControlPoints();
		bc.updateShape();
	}
	
	
	
	
	/**
	 * @return The SVG segment path list of the current Bézier curve.
	 * @since 2.0.0
	 */
	protected SVGPathSegList getPathSegList()
	{
		BezierCurve bc = (BezierCurve)shape;
		
		if(bc.getNbPoints()<2)
			return null;
		
		int size = bc.getNbPoints(), i;
		SVGPathSegList path = new SVGPathSegList();

		path.add(new SVGPathSegMoveto(bc.getPoint(0).x, bc.getPoint(0).y, false));
		path.add(new SVGPathSegCurvetoCubic(bc.getPoint(1).x, bc.getPoint(1).y, bc.getFirstControlPoint(0).x, 
					bc.getFirstControlPoint(0).y, bc.getFirstControlPoint(1).x, bc.getFirstControlPoint(1).y, false));
		
		for(i=2; i<size; i++)
			path.add(new SVGPathSegCurvetoCubic(bc.getPoint(i).x, bc.getPoint(i).y, 
												bc.getSecondControlPoint(i-1).x, bc.getSecondControlPoint(i-1).y, 
												bc.getFirstControlPoint(i).x, bc.getFirstControlPoint(i).y, false));
		
		if(!bc.isOpen())
		{
			if(bc.getCloseType()==BezierCurve.CLOSE_TYPE_CURVE)
			{
				LaTeXDrawPoint2D ctrl1b = bc.getFirstControlPoint(0).centralSymmetry(bc.getPoint(0));
				LaTeXDrawPoint2D ctrl2b = bc.getFirstControlPoint(-1).centralSymmetry(bc.getPoint(-1));
				
				path.add(new SVGPathSegCurvetoCubic(bc.getPoint(0).x, bc.getPoint(0).y, ctrl2b.x, ctrl2b.y, ctrl1b.x, ctrl1b.y, false));
			}
			
			path.add(new SVGPathSegClosePath());
		}

		return path;
	}

	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null || doc.getFirstChild().getDefs()==null)
			return null;

		BezierCurve bc = (BezierCurve)shape;
		SVGDefsElement defs = doc.getFirstChild().getDefs();
		SVGElement root = new SVGGElement(doc), elt;
		ArrowHead arrowHead1 = ((Arrowable)shape).getArrowHead1();
		ArrowHead arrowHead2 = ((Arrowable)shape).getArrowHead2();
		String path = getPathSegList().toString();
		SVGElement arrow1 = null, arrow2 = null, arrow1Shad = null, arrow2Shad = null;
		String arrow1Name = "arrow1-" + shape.getNumber();//$NON-NLS-1$
		String arrow2Name = "arrow2-" + shape.getNumber();//$NON-NLS-1$
		String arrow1ShadName = "arrow1Shad-" + shape.getNumber();//$NON-NLS-1$
		String arrow2ShadName = "arrow2Shad-" + shape.getNumber();//$NON-NLS-1$

		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_BEZIER_CURVE);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		
		if(!arrowHead1.isWithoutStyle())
		{
			arrow1     = new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, false);
			arrow1Shad = new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, true);

			arrow1.setAttribute(SVGAttributes.SVG_ID, arrow1Name);
			defs.appendChild(arrow1);

			if(shape.hasShadow())
			{
				arrow1Shad.setAttribute(SVGAttributes.SVG_ID, arrow1ShadName);
				defs.appendChild(arrow1Shad);
			}
		}

		if(!arrowHead2.isWithoutStyle() && bc.isOpen())
		{
			arrow2     = new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, false);
			arrow2Shad = new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, true);

			arrow2.setAttribute(SVGAttributes.SVG_ID, arrow2Name);
			defs.appendChild(arrow2);

			if(shape.hasShadow())
			{
				arrow2Shad.setAttribute(SVGAttributes.SVG_ID, arrow2ShadName);
				defs.appendChild(arrow2Shad);
			}
		}
       
       	if(shape.hasShadow())
		{
       		SVGElement shad = new SVGPathElement(doc);

			shad.setAttribute(SVGAttributes.SVG_D, path);
			setSVGShadowAttributes(shad, false);
			root.appendChild(shad);

			if(arrow1Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$

			if(arrow2Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$
		}
       	
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE) && shape.isFilled())
        {// The background of the borders must be filled is there is a shadow.
    		elt = new SVGPathElement(doc);
    		elt.setAttribute(SVGAttributes.SVG_D, path);
    		setSVGBorderBackground(elt, root);
        }
       
		elt = new SVGPathElement(doc);
		elt.setAttribute(SVGAttributes.SVG_D, path);
		root.appendChild(elt);
        
		if(shape.hasDoubleBoundary())
		{
			SVGElement dblBord = new SVGPathElement(doc);
			dblBord.setAttribute(SVGAttributes.SVG_D, path);
			setSVGDoubleBordersAttributes(dblBord);
			root.appendChild(dblBord);
		}
		
		setSVGAttributes(doc, elt, false);
		elt.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE +':'+ LaTeXDrawNamespace.XML_ROTATION, String.valueOf(shape.getRotationAngle()));

		if(arrow1 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1Name + ")");//$NON-NLS-1$ //$NON-NLS-2$

		if(arrow2 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2Name + ")");//$NON-NLS-1$ //$NON-NLS-2$

		if(bc.isShowPoints())
			root.appendChild(getShowPointsElement(doc));
		
		return root;
	}
	
	
	
	/**
	 * Creates an SVG g element that contains the 'show points' plotting.
	 * @param doc The owner document.
	 * @return The created g element or null if the shape has not the 'show points' option activated.
	 * @since 2.0.0
	 */
	protected SVGGElement getShowPointsElement(SVGDocument doc)
	{
		BezierCurve bc = (BezierCurve)shape;
		
		if(!bc.isShowPoints() || doc==null)
			return null;
		
		float blackDash  = bc.getBlackDashLength();
		float whiteDash  = bc.getWhiteDashLength();
		boolean hasDble  = bc.hasDoubleBoundary();
		Color col        = bc.getLinesColor();
		double doubleSep = bc.getDoubleSep();
		boolean isOpen   = bc.isOpen();
		int closeType    = bc.getCloseType();
		SVGGElement showPts  = new SVGGElement(doc);
		ArrowHead arrowHead1 = ((Arrowable)shape).getArrowHead1();
		ArrowHead arrowHead2 = ((Arrowable)shape).getArrowHead2();
		double thick = (hasDble ? bc.getDoubleSep()+bc.getThickness()*2. : bc.getThickness())/2.;
		double rad   = (arrowHead1.getDotSizeDim() + arrowHead1.getDotSizeNum()*thick*2.)/2.;
		int i = 0, size = bc.getNbPoints();
		
		showPts.setAttribute(new StringBuffer(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE).append(':').append(
											LaTeXDrawNamespace.XML_TYPE).toString(), LaTeXDrawNamespace.XML_TYPE_SHOW_PTS);
		
		/* Plotting the lines. */
		for(i=3; i<size; i+=2)
		{
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getPoint(i-1), bc.getSecondControlPoint(i-1),
								blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getSecondControlPoint(i-1), bc.getFirstControlPoint(i),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getFirstControlPoint(i), bc.getPoint(i),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
		}
		
		for(i=2; i<size; i+=2)
		{
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getPoint(i-1), bc.getSecondControlPoint(i-1),
								blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getSecondControlPoint(i-1), bc.getFirstControlPoint(i),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getFirstControlPoint(i), bc.getPoint(i),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
		}
		
		if(!isOpen && closeType==BezierCurve.CLOSE_TYPE_CURVE)
		{
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getPoint(-1), bc.getSecondControlPoint(-1),
								blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getSecondControlPoint(-1), bc.getSecondControlPoint(0),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
			showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getSecondControlPoint(0), bc.getPoint(0),
					  			blackDash, whiteDash, hasDble, 1, doubleSep));
		}
		
		showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getPoint(0), bc.getFirstControlPoint(0),
							blackDash, whiteDash, hasDble, 1, doubleSep));
		showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getFirstControlPoint(0), bc.getFirstControlPoint(1),
							blackDash, whiteDash, hasDble, 1, doubleSep));
		showPts.appendChild(getShowPointsLine(doc, (float) thick, col, bc.getFirstControlPoint(1), bc.getPoint(1),
							blackDash, whiteDash, hasDble, 1, doubleSep));
			
		// Plotting the dots.
		if(arrowHead1.isWithoutStyle() || (!isOpen && closeType==BezierCurve.CLOSE_TYPE_CURVE))
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getPoint(0), col));
		
		if(arrowHead2.isWithoutStyle() || (!isOpen && closeType==BezierCurve.CLOSE_TYPE_CURVE))
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getPoint(-1), col));
		
		for(i=1; i<size-1; i++)
		{
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getPoint(i), col));
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getSecondControlPoint(i), col));
		}
		
		for(i=0; i<size; i++)
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getFirstControlPoint(i), col));
		
		if(!isOpen && closeType==BezierCurve.CLOSE_TYPE_CURVE)
		{
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getSecondControlPoint(-1), col));
			showPts.appendChild(LShapeSVGGenerator.getShowPointsDot(doc, rad, bc.getSecondControlPoint(0), col));
		}
		
		return showPts;
	}
}
