/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package nsk.jdi.WatchpointRequest.addThreadFilter;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * WatchpointRequest.
 *
 * The test checks up on the following assertion:
 *     Restricts the events generated by this request
 *     to those in the given thread.
 * The cases to test include re-invocations of the method
 * addThreadFilter() on the same AccessWatchpointRequest object.
 * There are two AccessWatchpointRequests to check as follows:
 * (1) For AccessWatchpointRequest2, both invocations are with different
 * ThreadReferences restricting one AccessWatchpoint event to two threads.
 * The test expects no AccessWatchpoint event will be received.
 * (2) For AccessWatchpointRequest1, both invocations are with the same
 * ThreadReference restricting one AccessWatchpoint event to one thread.
 * The test expects this AccessWatchpoint event will be received.
 *
 * The test works as follows.
 * - The debugger resumes the debuggee and waits for the BreakpointEvent.
 * - The debuggee creates two threads, thread1 and thread2, and invokes
 *   the methodForCommunication to be suspended and
 *   to inform the debugger with the event.
 * - Upon getting the BreakpointEvent, the debugger
 *   - sets up AccessWatchpointRequests 1&2 within the method
 *     in the class addthreadfilter007aTestClass which will be calling by both threads,
 *   - restricts the AccessWatchpointRequest1 only to thread1,
 *   - restricts the AccessWatchpointRequest2 to both thread1 and thread2,
 *   - resumes debuggee's main thread, and
 *   - waits for the event.
 * - Debuggee's main thread starts both threads.
 * - Upon getting the event, the debugger performs the checks required.
 */

public class addthreadfilter007 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.WatchpointRequest.addThreadFilter.addthreadfilter007a";
        return new addthreadfilter007().runThis(argv, out);
    }

    private String testedClassName =
      "nsk.jdi.WatchpointRequest.addThreadFilter.addthreadfilter007aTestClass";


    protected void testRun() {
        if ( !vm.canWatchFieldAccess() ) {
            display("......vm.canWatchFieldAccess == false :: test cancelled");
            vm.exit(Consts.JCK_STATUS_BASE);
            return;
        }

        EventRequest eventRequest1 = null;
        EventRequest eventRequest2 = null;

        ThreadReference thread1 = null;
        ThreadReference thread2 = null;

        String thread1Name = "thread1";
        String thread2Name = "thread2";

        String property1 = "AccessWatchpointRequest1";
        String property2 = "AccessWatchpointRequest2";

        String fieldName = "var1";

        ReferenceType testClassReference = null;
        Event newEvent = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {
                case 0:
                testClassReference =
                     (ReferenceType) vm.classesByName(testedClassName).get(0);

                thread1 = (ThreadReference) debuggeeClass.getValue(
                                            debuggeeClass.fieldByName(thread1Name));
                thread2 = (ThreadReference) debuggeeClass.getValue(
                                            debuggeeClass.fieldByName(thread2Name));

                eventRequest1 = setting2AccessWatchpointRequest (thread1,
                                       testClassReference, fieldName,
                                       EventRequest.SUSPEND_ALL, property1);
                eventRequest2 = setting2AccessWatchpointRequest (thread1,
                                       testClassReference, fieldName,
                                       EventRequest.SUSPEND_ALL, property2);

                ((AccessWatchpointRequest) eventRequest1).addThreadFilter(thread1);
                ((AccessWatchpointRequest) eventRequest2).addThreadFilter(thread2);

                display("......waiting for AccessWatchpointEvent in tested thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1, eventRequest2}, waitTime, true);

                if (!(newEvent instanceof AccessWatchpointEvent)) {
                    setFailedStatus("ERROR: new event is not WatchpointEvent");
                } else {
                    display("       got expected event");

                    ThreadReference newEventThread = ((AccessWatchpointEvent) newEvent).thread();
                    String threadName = newEventThread.name();
                    display("       the event is in thread == " + newEventThread.name());
                    if ( !newEventThread.equals(thread1) ) {
                        setFailedStatus("ERROR: the event is not in thread1");
                    }
                }
                vm.resume();
                break;

                default:
                throw new Failure("** default case 1 **");
            }
        }
        return;
    }

    private AccessWatchpointRequest setting2AccessWatchpointRequest (
                                                  ThreadReference thread,
                                                  ReferenceType   testedClass,
                                                  String          fieldName,
                                                  int             suspendPolicy,
                                                  String          property        ) {
        try {
            display("......setting up AccessWatchpointRequest:");
            display("       thread: " + thread + "; class: " + testedClass + "; fieldName: " + fieldName);
            Field field = testedClass.fieldByName(fieldName);

            AccessWatchpointRequest
            awr = eventRManager.createAccessWatchpointRequest(field);
            awr.putProperty("number", property);

            if (thread != null)
                awr.addThreadFilter(thread);
            awr.setSuspendPolicy(suspendPolicy);

            display("      AccessWatchpointRequest has been set up");
            return awr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up AccessWatchpointRequest **");
        }
    }

}
