/*
 * Copyright (c) 2007, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @modules java.base/jdk.internal.misc:+open
 *
 * @summary converted from VM Testbase nsk/jdi/ClassPrepareRequest/addSourceNameFilter/addSourceNameFilter001.
 * VM Testbase keywords: [quick, jpda, jdi, feature_jdk6_jpda, vm6]
 * VM Testbase readme:
 * DESCRIPTION
 *     The test checks up that a result of the method com.sun.jdi.ClassPrepareRequest.addSourceNameFilter(String sourceNamePattern)
 *     complies with its spec:
 *     "Restricts the events generated by this request to the preparation of reference types for which the
 *     restricted regular expression 'sourceNamePattern' matches one of the 'sourceNames' for the reference type
 *     being prepared."
 *     Test scenario:
 *     Debugger VM create ClassPrepareEventRequest, add source name filter and force debuggee load class. When debuggee
 *     finish class loading debugger checks is ClassPrepareEvent was correct filtered or not.
 *     Following source name filters are used:
 *         - exact source name
 *         - invalid source name (don't expect events with this filter)
 *         - pattern starting with '*'
 *         - pattern ending with '*'
 *
 * @library /vmTestbase
 *          /test/lib
 *
 * @build nsk.jdi.ClassPrepareRequest.addSourceNameFilter.addSourceNameFilter001.addSourceNameFilter001
 *        nsk.jdi.ClassPrepareRequest.addSourceNameFilter.addSourceNameFilter001.TestClass2
 *        nsk.share.jdi.TestClass1
 * @run driver jdk.test.lib.FileInstaller . .
 * @run main/othervm PropertyResolvingWrapper
 *      nsk.jdi.ClassPrepareRequest.addSourceNameFilter.addSourceNameFilter001.addSourceNameFilter001
 *      -verbose
 *      -arch=${os.family}-${os.simpleArch}
 *      -waittime=5
 *      -debugee.vmkind=java
 *      -transport.address=dynamic
 *      "-debugee.vmkeys=${test.vm.opts} ${test.java.opts}"
 *      -testClassPath ${test.class.path}
 */

package nsk.jdi.ClassPrepareRequest.addSourceNameFilter.addSourceNameFilter001;

import java.io.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;
import nsk.share.Consts;
import nsk.share.TestBug;
import nsk.share.jdi.*;
import nsk.share.jpda.AbstractDebuggeeTest;
import java.util.*;

public class addSourceNameFilter001 extends TestDebuggerType2 {
    public static void main(String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run(String argv[], PrintStream out) {
        return new addSourceNameFilter001().runIt(argv, out);
    }

    protected boolean canRunTest() {
        if (!vm.canUseSourceNameFilters()) {
            log.display("TEST CANCELLED due to:  vm.canUseSourceNameFilters() = false");
            return false;
        } else
            return true;
    }

    protected String debuggeeClassName() {
        if (classpath == null)
            throw new TestBug("Debugger requires 'testClassPath' parameter");

        return AbstractJDIDebuggee.class.getName() + " -testClassPath " + classpath;
    }

    // listener counting ClassPrepareEvent
    public class ClassPrepareEventListener extends EventHandler.EventListener {

        private boolean eventReceived;

        private String expectedClassName;

        ClassPrepareEventListener(String expectedClassName) {
            this.expectedClassName = expectedClassName;
        }

        public boolean eventReceived(Event event) {
            if (event instanceof ClassPrepareEvent) {
                ClassPrepareEvent classPrepareEvent = (ClassPrepareEvent) event;
                log.display("Received ClassPrepareEvent: '" + event + "'. Event for class: " + classPrepareEvent.referenceType().name());

                if (eventReceived) {
                    setSuccess(false);
                    log.complain("Only 1 ClassPrepareEvent is expected for tested request");
                } else {
                    if (!classPrepareEvent.referenceType().name().equals(expectedClassName)) {
                        setSuccess(false);
                        log.complain("Received event for wrong class, expected class is " + expectedClassName);
                    }
                    eventReceived = true;
                }

                vm.resume();
                return true;
            }

            return false;
        }
    }

    private EventHandler eventHandler;

    // force debuggee load class and check is ClassPrepareEvent was generated
    private void testSourceFilter(String expectedClassName,
            List<String> classesToLoad,
            String sourceName,
            boolean expectEvent) {
        ClassPrepareRequest request;

        // create request with filter
        request = debuggee.getEventRequestManager().createClassPrepareRequest();
        request.setSuspendPolicy(EventRequest.SUSPEND_ALL);
        request.addSourceNameFilter(sourceName);
        request.enable();

        // this listener waits for ClassPrepareEvent
        ClassPrepareEventListener listener = new ClassPrepareEventListener(expectedClassName);
        eventHandler.addListener(listener);

        for (String className : classesToLoad) {
            log.display("Loading class: " + className + ", using following source filter: " + sourceName);

            // force debuggee load class
            pipe.println(AbstractDebuggeeTest.COMMAND_LOAD_CLASS + ":" + className);

            if (!isDebuggeeReady())
                return;
        }

        request.disable();

        eventHandler.removeListener(listener);

        // check is event was correctly filtered or not
        if (expectEvent) {
            if (!listener.eventReceived) {
                setSuccess(false);
                log.complain("Expected ClassPrepareEvent was not received");
            }
        } else {
            if (listener.eventReceived) {
                setSuccess(false);
                log.complain("Unexpected ClassPrepareEvent was received");
            }
        }
    }

    public void doTest() {
        eventHandler = new EventHandler(debuggee, log);
        eventHandler.startListening();

        String className = "nsk.share.jdi.TestClass1";
        String sourceName = "TestClass1.java";

        List<String> classesToLoad = new ArrayList<String>();
        classesToLoad.add(className);
        // don't expect events for TestClass2
        classesToLoad.add(TestClass2.class.getName());

        // set invalid source name, don't expect events
        testSourceFilter(className, classesToLoad, sourceName + "_InvalidSourceName", false);

        // set valid source name
        testSourceFilter(className, classesToLoad, sourceName, true);

        sourceName = "TestClass1.*";
        // use pattern filter ending with '*'
        testSourceFilter(className, classesToLoad, sourceName, true);

        sourceName = "*Class1.java";
        // use pattern filter starting with '*'
        testSourceFilter(className, classesToLoad, sourceName, true);

        eventHandler.stopEventHandler();
    }
}
