# frozen_string_literal: true

require 'delegate'

module Rack
  class Attack
    module StoreProxy
      class RedisCacheStoreProxy < SimpleDelegator
        def self.handle?(store)
          store.class.name == "ActiveSupport::Cache::RedisCacheStore"
        end

        def increment(name, amount = 1, options = {})
          # RedisCacheStore#increment ignores options[:expires_in].
          #
          # So in order to workaround this we use RedisCacheStore#write (which sets expiration) to initialize
          # the counter. After that we continue using the original RedisCacheStore#increment.
          rescuing do
            if options[:expires_in] && !read(name)
              write(name, amount, options)

              amount
            else
              super
            end
          end
        end

        def read(*_args)
          rescuing { super }
        end

        def write(name, value, options = {})
          rescuing do
            super(name, value, options.merge!(raw: true))
          end
        end

        private

        def rescuing
          yield
        rescue Redis::BaseError
          nil
        end
      end
    end
  end
end
