/*
 * Copyright © 2014 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/**
 * \file invalid-primitive.c
 *
 * From the ARB_tessellation_shader spec (Errors section):
 *
 *  "The error INVALID_OPERATION is generated by Begin (or vertex array
 *  commands that implicitly call Begin) if the active program contains a
 *  tessellation control or evaluation shader and the primitive mode is not
 *  PATCHES.
 *
 *  The error INVALID_OPERATION is generated by Begin (or vertex array
 *  commands that implicitly call Begin) if the primitive mode is PATCHES if
 *  either:
 *
 *    *  the active program contains no tessellation evaluation shader, or
 *    *  no program is active."
 *
 * Test for all those errors except without active program.
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 32;
	config.supports_gl_core_version = 32;

PIGLIT_GL_TEST_CONFIG_END


static const char *const vs_source =
"#version 150\n"
"void main() { gl_Position = vec4(0.0); }\n";

static const char *const tcs_source =
"#version 150\n"
"#extension GL_ARB_tessellation_shader: require\n"
"layout(vertices = 3) out;\n"
"void main() { gl_out[gl_InvocationID].gl_Position = vec4(0.0); }\n";

static const char *const tes_source =
"#version 150\n"
"#extension GL_ARB_tessellation_shader: require\n"
"layout(triangles) in;\n"
"void main() { gl_Position = vec4(0.0); }\n";

static const char *const fs_source =
"#version 150\n"
"void main() { gl_FragColor = vec4(0.0); }\n";


void
piglit_init(int argc, char **argv)
{
	bool pass = true;
	int i;
	unsigned int vao, tess_prog, normal_prog;
	static const GLenum primitive[] = {
		GL_POINTS,
		GL_LINES,
		GL_LINE_LOOP,
		GL_LINE_STRIP,
		GL_TRIANGLES,
		GL_TRIANGLE_STRIP,
		GL_TRIANGLE_FAN,
		GL_LINES_ADJACENCY,
		GL_LINE_STRIP_ADJACENCY,
		GL_TRIANGLES_ADJACENCY,
		GL_TRIANGLE_STRIP_ADJACENCY,
	};

	piglit_require_extension("GL_ARB_tessellation_shader");

	glGenVertexArrays(1, &vao);
	glBindVertexArray(vao);

	tess_prog = piglit_build_simple_program_multiple_shaders(
			GL_VERTEX_SHADER, vs_source,
			GL_TESS_CONTROL_SHADER, tcs_source,
			GL_TESS_EVALUATION_SHADER, tes_source,
			GL_FRAGMENT_SHADER, fs_source,
			0);
	normal_prog = piglit_build_simple_program_multiple_shaders(
			GL_VERTEX_SHADER, vs_source,
			GL_FRAGMENT_SHADER, fs_source,
			0);

	for (i = 0; i < ARRAY_SIZE(primitive); ++i) {
		glUseProgram(tess_prog);
		pass = piglit_check_gl_error(GL_NO_ERROR) && pass;
		glDrawArrays(primitive[i], 0, 12);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;
	}

	/* NV_gpu_shader5 allows drawing patches without tessellation
	 * evaluation shader.
	 */
	if (!piglit_is_extension_supported("GL_NV_gpu_shader5")) {
		glUseProgram(normal_prog);
		pass = piglit_check_gl_error(GL_NO_ERROR) && pass;
		glDrawArrays(GL_PATCHES, 0, 12);
		pass = piglit_check_gl_error(GL_INVALID_OPERATION) && pass;
	}

	glUseProgram(0);
	glDeleteProgram(tess_prog);
	glDeleteProgram(normal_prog);
	glBindVertexArray(0);
	glDeleteVertexArrays(1, &vao);

	pass = piglit_check_gl_error(GL_NO_ERROR) && pass;

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}


enum piglit_result
piglit_display(void)
{
	return PIGLIT_PASS;
}

