/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "Bad-words IMSpector filter plugin"
#define PLUGIN_SHORT_NAME "Bad-words"

#define DEFAULT_REPLACE_CHAR '*'

extern "C"
{
	bool initfilterplugin(struct filterplugininfo &filterplugininfo,
		class Options &options, bool debugmode);
	void closefilterplugin(void);
	bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent);
};

int readbadwords(std::string filename);

std::vector<std::string> badwords;
char replacecharacter = DEFAULT_REPLACE_CHAR;
int blockcount = 0;
bool localdebugmode = false;

bool initfilterplugin(struct filterplugininfo &filterplugininfo,
	class Options &options, bool debugmode)
{
	std::string badwordsfilename = options["badwords_filename"];
	std::string badwordsreplacecharacter = options["badwords_replace_character"];
	std::string badwordsblockcount = options["badwords_block_count"];
	
	if (badwordsfilename.empty()) return false;

	localdebugmode = debugmode;
	
	int count = readbadwords(badwordsfilename);

	if (count == -1)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't open bad words file %s",
			badwordsfilename.c_str());
		return false;
	}
		
	if (!badwordsreplacecharacter.empty())
		replacecharacter = badwordsreplacecharacter[0];
	if (!badwordsblockcount.empty())
		blockcount = atol(badwordsblockcount.c_str());
		
	syslog(LOG_INFO, PLUGIN_SHORT_NAME ": Loaded %d bad-words, replacing with '%c' and blocking at %d",
		count, replacecharacter, blockcount);

	filterplugininfo.pluginname = PLUGIN_NAME;
	
	return true;
}

void closefilterplugin(void)
{
	return;
}

/* The main plugin function. See filterplugin.cpp. */
bool filter(char *originalbuffer, char *modifiedbuffer, struct imevent &imevent)
{
	int count = 0;
	
	/* We may have no text to filter. */
	if (!strlen(originalbuffer)) return false;

	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": filtering before: original: %s modified: %s",
		originalbuffer, modifiedbuffer);

	for (std::vector<std::string>::iterator i = badwords.begin();
		i != badwords.end(); i++)
	{
		const char *needle = (*i).c_str();
		size_t needlelength = (*i).length();
		
		char *s = modifiedbuffer;
		
		while ((s = strcasestr(s, needle)))
		{
			if (s > modifiedbuffer)
			{
				if (isalpha(*(s - 1)) && isalpha(*(s + needlelength)))
				{
					s++;
					continue;
				}
			}
		
			count++;
			
			memset(s, (int) replacecharacter, needlelength);
		}
	}
	
	debugprint(localdebugmode, PLUGIN_SHORT_NAME ": filtering after: modified: %s count: %d (limit: %d)",
		modifiedbuffer, count, blockcount);
		
	if (count) imevent.categories += stringprintf("%d badwords;", count);
	
	return blockcount ? count >= blockcount : false;
}

/* Reads a file into the badwords list. */
int readbadwords(std::string filename)
{
	FILE *hfile = NULL;
	char buffer[STRING_SIZE];
	int result = 0;
	
	memset(buffer, 0, STRING_SIZE);
				
	if (!(hfile = fopen(filename.c_str(), "r")))
	{
		syslog(LOG_ERR, "Error: Unable to open badwords list");
		return -1;
	}
		
	while (fgets(buffer, STRING_SIZE, hfile))
	{
		stripnewline(buffer);
		
		if (!strlen(buffer)) break;
		
		badwords.push_back(buffer);
		result++;
	}
	
	fclose(hfile);

	return result;
}
