/* -*- c++ -*- */
/*
 * Copyright 2013 Dimitri Stolnikov <horiz0n@gmx.net>
 *
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdexcept>
#include <iostream>
#include <algorithm>

#include <boost/assign.hpp>
#include <boost/format.hpp>
#include <boost/detail/endian.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/thread/thread.hpp>

#include <gnuradio/io_signature.h>

#include "airspyhf_source_c.h"
#include "arg_helpers.h"

using namespace boost::assign;

#define AIRSPYHF_FORMAT_ERROR(ret, msg) \
  boost::str( boost::format(msg " (%1%)") % ret )

#define AIRSPYHF_THROW_ON_ERROR(ret, msg) \
  if ( ret != AIRSPYHF_SUCCESS ) \
  { \
    throw std::runtime_error( AIRSPYHF_FORMAT_ERROR(ret, msg) ); \
  }

#define AIRSPYHF_FUNC_STR(func, arg) \
  boost::str(boost::format(func "(%1%)") % arg) + " has failed"

airspyhf_source_c_sptr make_airspyhf_source_c (const std::string & args)
{
  return gnuradio::get_initial_sptr(new airspyhf_source_c (args));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 0 input and 1 output.
 */
static const int MIN_IN = 0;	// mininum number of input streams
static const int MAX_IN = 0;	// maximum number of input streams
static const int MIN_OUT = 1;	// minimum number of output streams
static const int MAX_OUT = 1;	// maximum number of output streams

/*
 * The private constructor
 */
airspyhf_source_c::airspyhf_source_c (const std::string &args)
  : gr::sync_block ("airspyhf_source_c",
        gr::io_signature::make(MIN_IN, MAX_IN, sizeof (gr_complex)),
        gr::io_signature::make(MIN_OUT, MAX_OUT, sizeof (gr_complex))),
    _dev(NULL),
    _sample_rate(0),
    _center_freq(0),
    _freq_corr(0)
{
  int ret;

  dict_t dict = params_to_dict(args);

  _dev = NULL;
  ret = airspyhf_open( &_dev );
  AIRSPYHF_THROW_ON_ERROR(ret, "Failed to open Airspy HF+ device")

  uint32_t num_rates;
  airspyhf_get_samplerates(_dev, &num_rates, 0);
  uint32_t *samplerates = (uint32_t *) malloc(num_rates * sizeof(uint32_t));
  airspyhf_get_samplerates(_dev, samplerates, num_rates);
  for (size_t i = 0; i < num_rates; i++)
    _sample_rates.push_back( std::pair<double, uint32_t>( samplerates[i], i ) );
  free(samplerates);

  /* since they may (and will) give us an unsorted array we have to sort it here
   * to play nice with the monotonic requirement of meta-range later on */
  std::sort(_sample_rates.begin(), _sample_rates.end());

  std::cerr << "Using libairspyhf" << AIRSPYHF_VERSION << ", samplerates: ";

  for (size_t i = 0; i < _sample_rates.size(); i++)
    std::cerr << boost::format("%gM ") % (_sample_rates[i].first / 1e6);

  std::cerr << std::endl;

  set_center_freq( (get_freq_range().start() + get_freq_range().stop()) / 2.0 );
  set_sample_rate( get_sample_rates().start() );

  _fifo = new boost::circular_buffer<gr_complex>(5000000);
  if (!_fifo) {
    throw std::runtime_error( std::string(__FUNCTION__) + " " +
                              "Failed to allocate a sample FIFO!" );
  }
}

/*
 * Our virtual destructor.
 */
airspyhf_source_c::~airspyhf_source_c ()
{
  int ret;

  if (_dev) {
    if ( airspyhf_is_streaming( _dev ) )
    {
      ret = airspyhf_stop( _dev );
      if ( ret != AIRSPYHF_SUCCESS )
      {
        std::cerr << AIRSPYHF_FORMAT_ERROR(ret, "Failed to stop RX streaming") << std::endl;
      }
    }

    ret = airspyhf_close( _dev );
    if ( ret != AIRSPYHF_SUCCESS )
    {
      std::cerr << AIRSPYHF_FORMAT_ERROR(ret, "Failed to close AirSpy") << std::endl;
    }
    _dev = NULL;
  }

  if (_fifo)
  {
    delete _fifo;
    _fifo = NULL;
  }
}

int airspyhf_source_c::_airspyhf_rx_callback(airspyhf_transfer_t *transfer)
{
  airspyhf_source_c *obj = (airspyhf_source_c *)transfer->ctx;

  return obj->airspyhf_rx_callback((float *)transfer->samples, transfer->sample_count);
}

int airspyhf_source_c::airspyhf_rx_callback(void *samples, int sample_count)
{
  size_t i, n_avail, to_copy, num_samples = sample_count;
  float *sample = (float *)samples;

  _fifo_lock.lock();

  n_avail = _fifo->capacity() - _fifo->size();
  to_copy = (n_avail < num_samples ? n_avail : num_samples);

  for (i = 0; i < to_copy; i++ )
  {
    /* Push sample to the fifo */
    _fifo->push_back( gr_complex( *sample, *(sample+1) ) );

    /* offset to the next I+Q sample */
    sample += 2;
  }

  _fifo_lock.unlock();

  /* We have made some new samples available to the consumer in work() */
  if (to_copy) {
    //std::cerr << "+" << std::flush;
    _samp_avail.notify_one();
  }

  /* Indicate overrun, if neccesary */
  if (to_copy < num_samples)
    std::cerr << "O" << std::flush;

  return 0; // TODO: return -1 on error/stop
}

bool airspyhf_source_c::start()
{
  if ( ! _dev )
    return false;

  int ret = airspyhf_start( _dev, _airspyhf_rx_callback, (void *)this );
  if ( ret != AIRSPYHF_SUCCESS ) {
    std::cerr << "Failed to start RX streaming (" << ret << ")" << std::endl;
    return false;
  }

  return true;
}

bool airspyhf_source_c::stop()
{
  if ( ! _dev )
    return false;

  int ret = airspyhf_stop( _dev );
  if ( ret != AIRSPYHF_SUCCESS ) {
    std::cerr << "Failed to stop RX streaming (" << ret << ")" << std::endl;
    return false;
  }

  return true;
}

int airspyhf_source_c::work( int noutput_items,
                        gr_vector_const_void_star &input_items,
                        gr_vector_void_star &output_items )
{
  gr_complex *out = (gr_complex *)output_items[0];

  bool running = false;

  if ( _dev )
    running = airspyhf_is_streaming( _dev );

  if ( ! running )
    return WORK_DONE;

  boost::unique_lock<boost::mutex> lock(_fifo_lock);

  /* Wait until we have the requested number of samples */
  int n_samples_avail = _fifo->size();

  while (n_samples_avail < noutput_items) {
    _samp_avail.wait(lock);
    n_samples_avail = _fifo->size();
  }

  for(int i = 0; i < noutput_items; ++i) {
    out[i] = _fifo->at(0);
    _fifo->pop_front();
  }

  return noutput_items;
}

std::vector<std::string> airspyhf_source_c::get_devices()
{
  std::vector<std::string> devices;
  std::string label;

  int ret;
  airspyhf_device *dev = NULL;
  ret = airspyhf_open(&dev);
  if ( AIRSPYHF_SUCCESS == ret )
  {
    std::string args = "airspyhf=0,label='AirspyHF'";
    devices.push_back( args );
    ret = airspyhf_close(dev);
  }

  return devices;
}

size_t airspyhf_source_c::get_num_channels()
{
  return 1;
}

osmosdr::meta_range_t airspyhf_source_c::get_sample_rates()
{
  osmosdr::meta_range_t range;

  for (size_t i = 0; i < _sample_rates.size(); i++)
    range += osmosdr::range_t( _sample_rates[i].first );

  return range;
}

double airspyhf_source_c::set_sample_rate( double rate )
{
  int ret = AIRSPYHF_SUCCESS;

  if (_dev) {
    bool found_supported_rate = false;
    uint32_t samp_rate_index = 0;

    for( unsigned int i = 0; i < _sample_rates.size(); i++ )
    {
      if( _sample_rates[i].first == rate )
      {
        samp_rate_index = _sample_rates[i].second;

        found_supported_rate = true;
      }
    }

    if ( ! found_supported_rate )
    {
      throw std::runtime_error(
        boost::str( boost::format("Unsupported samplerate: %gM") % (rate/1e6) ) );
    }

    ret = airspyhf_set_samplerate( _dev, samp_rate_index );
    if ( AIRSPYHF_SUCCESS == ret ) {
      _sample_rate = rate;
    } else {
      AIRSPYHF_THROW_ON_ERROR( ret, AIRSPYHF_FUNC_STR( "airspyhf_set_samplerate", rate ) )
    }
  }

  return get_sample_rate();
}

double airspyhf_source_c::get_sample_rate()
{
  return _sample_rate;
}

osmosdr::freq_range_t airspyhf_source_c::get_freq_range( size_t chan )
{
  osmosdr::freq_range_t range;

  range += osmosdr::range_t( 0.0, 260.0e6 );

  return range;
}

double airspyhf_source_c::set_center_freq( double freq, size_t chan )
{
  int ret;

  if (_dev) {
    ret = airspyhf_set_freq( _dev, freq );
    if ( AIRSPYHF_SUCCESS == ret ) {
      _center_freq = freq;
    } else {
      AIRSPYHF_THROW_ON_ERROR( ret, AIRSPYHF_FUNC_STR( "airspyhf_set_freq", freq ) )
    }
  }

  return get_center_freq( chan );
}

double airspyhf_source_c::get_center_freq( size_t chan )
{
  return _center_freq;
}

double airspyhf_source_c::set_freq_corr( double ppm, size_t chan )
{
  int ret;
  int32_t ppb = (int32_t) (ppm * 1.0e3);

  if (_dev) {
    ret = airspyhf_set_calibration( _dev, ppb );
    if ( AIRSPYHF_SUCCESS == ret ) {
      _freq_corr = ppm;
    } else {
      AIRSPYHF_THROW_ON_ERROR( ret, AIRSPYHF_FUNC_STR( "airspyhf_set_calibration", ppm ) )
    }
  }

  return ppm;
}

double airspyhf_source_c::get_freq_corr( size_t chan )
{
  return _freq_corr;
}

std::vector<std::string> airspyhf_source_c::get_gain_names( size_t chan )
{
  return {};
}

osmosdr::gain_range_t airspyhf_source_c::get_gain_range( size_t chan )
{
  return osmosdr::gain_range_t();
}

osmosdr::gain_range_t airspyhf_source_c::get_gain_range( const std::string & name, size_t chan )
{
  return osmosdr::gain_range_t();
}


double airspyhf_source_c::set_gain( double gain, size_t chan )
{
  return gain;
}

double airspyhf_source_c::set_gain( double gain, const std::string & name, size_t chan)
{
  return gain;
}

double airspyhf_source_c::get_gain( size_t chan )
{
  return 0.0;
}

double airspyhf_source_c::get_gain( const std::string & name, size_t chan )
{
  return 0.0;
}

std::vector< std::string > airspyhf_source_c::get_antennas( size_t chan )
{
  std::vector< std::string > antennas;

  antennas += get_antenna( chan );

  return antennas;
}

std::string airspyhf_source_c::set_antenna( const std::string & antenna, size_t chan )
{
  return get_antenna( chan );
}

std::string airspyhf_source_c::get_antenna( size_t chan )
{
  return "RX";
}
