/* $Id: graphics.c,v 1.98 2003/06/16 11:46:17 sjoerd Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <SDL.h>
#include <SDL_ttf.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <assert.h>

#include "graphics.h"
#include "list.h"
#include "global.h"
#include "svg.h"
#include "sprite.h"
#include "player.h"

#define GDEBUG(...) DEBUG(DGRAPHICS,"Graphics",__VA_ARGS__)

#define SCOREBOARD_WIDTH 100

#ifndef SCALE_X
#define SCALE_X 1000
#endif

#ifndef SCALE_Y
#define SCALE_Y 1000
#endif

#define FONTNAME "bitstream-vera.ttf"
/* Used for static string buffers */
#define MAXSTRLEN 256
/******************************************************************************
 * Private functions
 *****************************************************************************/

Vector *graphics_scale(Graphics *g, Vector *v) {
	Vector *w;
	w = vector_mul(v, g->ratio);
	w->x = w->x / SCALE_X;
	w->y = w->y / SCALE_Y;
	return w;
}
/*** Graphics ***/

void
graphics_add_rect_update(Graphics *g,SDL_Rect rect) {
  if (g->update_number == g->update_queue_len) {
    g->update_queue_len += 20;
    g->update_rects_queue = realloc(g->update_rects_queue,
                                    g->update_queue_len * sizeof(SDL_Rect));
  }

  g->update_rects_queue[g->update_number] = rect;
  g->update_number++;
}

/*** Graphic sprite cache ***/
Graphics_cache *
new_graphics_cache(int type,Vector *size, Graphics_sprite_source *svg) {
  int x;
  Graphics_cache *result;
  result = malloc(sizeof(Graphics_cache));
  assert(result != NULL);
  result->type = type;
  result->size = copy_vector(size);
  for (x=0;x < SS_NRSTATES; x++) {
    result->animations[x] = NULL;
  }
  result->svg = svg;
  return result;
}

void 
graphics_cache_change_svg(Graphics_cache *gc, Graphics_sprite_source *svg) {
  int x;
  for (x = 0 ; x < SS_NRSTATES; x++) {
    if (gc->animations[x] != NULL)
      del_graphics_sprite_anim(gc->animations[x]);
    gc->animations[x] = NULL;
  }
  gc->svg = svg;
}

void 
del_graphics_cache(Graphics_cache *gc) {
  int x;
  del_vector(gc->size);
  for (x = 0; x < SS_NRSTATES ; x++) {
    if (gc->animations[x] != NULL)
      del_graphics_sprite_anim(gc->animations[x]);
  }
  free(gc);
}

SDL_Surface *
graphics_cache_getsprite(Graphics_cache *gc,Graphics *g,
                         Sprite_state state,int *framenr) {
  if (gc->animations[state] == NULL) { 
     GDEBUG("Requesting animation for state: %d of size %dx%d",
                 state,gc->size->x,gc->size->y);

    gc->animations[state] = draw_sprite_anim(g->screen,gc->size,
                                         state, gc->svg);
  }
  return sprite_anim_getsurface(gc->animations[state],
                                state,framenr);
}

SDL_Surface *
graphics_cache_getnext_sprite(Graphics_cache *gc, Graphics *g, 
                              Graphics_object *go, Model_object *mo) {
  int spritestate = 0;

  if (IS_FOOD(mo)) {
    if (IS_POWEREDUP(mo)) {
      spritestate = SS_FOOD_UPGRADE;
    } else {
      spritestate = SS_FOOD_NORMAL;
    }
  } else if (IS_PLAYER(mo)) {
    /* player */
    if (IS_DEAD(mo)) {
      spritestate = SS_PLAYER_DEAD;
    } else if (mo->events & EVF_HIT) { 
      spritestate = SS_PLAYER_HIT;
    } else if (IS_UNCONSCIOUS(mo)) {
      spritestate = SS_PLAYER_UNCONSCIOUS;
    } else if (mo->events & EVF_THROW) {
      spritestate = (LOOKS_LEFT(mo) ? SS_PLAYER_LTHROW : SS_PLAYER_RTHROW);
    } else if (!IS_STANDING(mo)) {
      spritestate = (LOOKS_LEFT(mo) ? SS_PLAYER_LJUMP : SS_PLAYER_RJUMP);
    } else if (mo->speed->x != 0) {
      spritestate = (LOOKS_LEFT(mo) ? SS_PLAYER_LWALK : SS_PLAYER_RWALK);
    } else {
      spritestate = LOOKS_LEFT(mo) ? SS_PLAYER_LSTAND : SS_PLAYER_RSTAND;
    }
  } else {
    spritestate = SS_SCENERY_NORMAL;
  }
  if (go->state != spritestate) {
    go->state = spritestate;
    go->frame = 0;
  } else {
    go->frame++;
  }
  return graphics_cache_getsprite(gc,g,spritestate,&(go->frame));
}
/*** Graphic drawing info ***/
Graphics_drawinfo *
new_graphics_drawinfo(void) {
  Graphics_drawinfo *result;
  result = malloc(sizeof(Graphics_drawinfo));
  assert(result != NULL);
  /* initialize everything to 0 */
  memset(result,0,sizeof(Graphics_drawinfo));
  result->sprite = NULL;
  return result;
}

void
del_graphics_drawinfo(Graphics_drawinfo *gd) {
  free(gd);
}

void 
graphics_drawinfo_setsprite(Graphics_drawinfo *gd,SDL_Surface *sprite) {
  if (sprite == gd->sprite) {
    gd->flags &= ~(Graphics_SPRITECHANGED);
  } else {
    gd->flags |= Graphics_SPRITECHANGED;
  }
  gd->sprite = sprite;
}

void
graphics_drawinfo_set_pos(Graphics_drawinfo *gd,Graphics *g, 
                          Vector *pos,Vector *size) {
  SDL_Rect rect;
  int overlap;

  rect = gd->prev;
  gd->prev = gd->cur;
  gd->cur = rect;

  gd->cur.x = (pos->x * g->ratio->x)/SCALE_X + g->goffset->x;
  gd->cur.y = (pos->y * g->ratio->y)/SCALE_Y + g->goffset->y;
  gd->cur.w = size->x;
  gd->cur.h = size->y;

  gd->blitrect = gd->cur;
  gd->blitrect.y = gd->blitrect.x = 0;  

  /* corrections */
  if (gd->cur.x < g->goffset->x) {
    overlap = g->goffset->x - gd->cur.x;
    gd->blitrect.x = MIN(gd->blitrect.w,overlap);
    gd->blitrect.w = MAX(0,gd->blitrect.w - overlap);
    gd->cur.w = MAX(0,gd->cur.w - overlap);
    gd->cur.x = g->goffset->x;
  }
  if (gd->cur.x + gd->cur.w > g->goffset->x + g->gsize->x) {
    overlap = gd->cur.x + gd->cur.w - g->goffset->x - g->gsize->x;
    gd->blitrect.w = MAX(0,gd->blitrect.w - overlap);
    gd->cur.w = MAX(0,gd->cur.w - overlap);
  }
  if (gd->cur.y < g->goffset->y) {
    overlap = g->goffset->y - gd->cur.y;
    gd->blitrect.y = MIN(gd->blitrect.h,overlap);
    gd->blitrect.h = MAX(0,gd->blitrect.h - overlap);
    gd->cur.w = MAX(0,gd->cur.h - overlap);
    gd->cur.y = g->goffset->y;
  }
  if (gd->cur.y + gd->cur.h > g->goffset->y + g->gsize->y) {
    overlap = gd->cur.y + gd->cur.y - g->goffset->y - g->gsize->y;
    gd->blitrect.h = MAX(0,gd->blitrect.h - overlap);
    gd->cur.h = MAX(0,gd->cur.h - overlap);
  }
  if (gd->prev.x == gd->cur.x && 
      gd->cur.y == gd->prev.y &&
      gd->cur.w == gd->prev.w &&
      gd->cur.h == gd->prev.h) {
    gd->flags &= ~(Graphics_POSCHANGED);
    return;
  }

  gd->flags |= Graphics_POSCHANGED;
}

/*** Graphic object ***/
int
find_scache_with_size(void *data, void *user_data) {
  Graphics_cache *c = (Graphics_cache *) data;
  Vector *size = ((Vector *)user_data);
  return vectors_equal(c->size,size);
}

Graphics_cache *
find_scache(Graphics *g,int type, Vector *size) {
  List *gcl = g->sprites[type];
  Graphics_cache *c;

  c = list_search(gcl,find_scache_with_size,size);

  if (c == NULL) {
    GDEBUG("Creating new graphic cache type: %d size: %dx%d",type,
        size->x,size->y);
    c = new_graphics_cache(type,size,g->svg[type]);
    list_append(gcl,c);
  }
  return c;
}

Graphics_object *
new_graphics_object(Graphics *g, Model_object *mo) {
  Graphics_object *result;
  result = malloc(sizeof(Graphics_object));
  assert(result != NULL);
	result->private_graphic = FALSE;
  result->frame = 0;
  /* init drawing info */
  result->dinfo = new_graphics_drawinfo();

  if (IS_FOOD(mo)) result->dinfo->layer = GRAPHICS_LAYER_FRONT;
  else if (IS_PLAYER(mo)) result->dinfo->layer = GRAPHICS_LAYER_MIDDLE;
  else if (IS_SCENERY(mo)) result->dinfo->layer = GRAPHICS_LAYER_BACK;

  list_append(g->layers[result->dinfo->layer],result->dinfo);
  /* init sprite info */
  result->size = graphics_scale(g,mo->size);
  result->scache = find_scache(g,mo->type,result->size);
	result->pinfo = NULL;
  result->state = -1;
  return result;
}

void
del_graphics_object(Graphics_object *go, Graphics *g) {
  del_vector(go->size);
  if (go->private_graphic) { 
    del_sprite_source(go->scache->svg);
    del_graphics_cache(go->scache);
  } 
  list_del(g->layers[go->dinfo->layer],go->dinfo);
  del_graphics_drawinfo(go->dinfo);
  free(go);
}

/*** Graphics player info object **/
Graphics_player_info *
new_graphics_player_info(char *name, int islocal) {
	Graphics_player_info *result ;
	int i;

  result = malloc(sizeof(Graphics_player_info));
  assert(result != NULL);

	result->name = strdup(name);
	result->score = 0;
	result->life = 0;
	result->maxlife = 0;
	result->is_local = islocal;
  result->place = new_vector(0,0);
  result->stack_top = -1;
  result->stacklength = 0;
  result->flags = 0;
  result->is_upgraded = FALSE;
	for (i = 0; i < OT_NRFOOD; i++)
		result->foodcaches[i] = NULL;

  result->name_graphic = NULL;

	return result;
}

void 
del_graphics_player_info(Graphics_player_info *pinfo) {
  int x;
  free(pinfo->name);
  del_vector(pinfo->place);
  for (x = 0 ; x < OT_NRFOOD; x++) {
    if (pinfo->foodcaches[x] != NULL) {
      del_graphics_cache(pinfo->foodcaches[x]);
    }
  }
  if (pinfo->name_graphic != NULL) {
    SDL_FreeSurface(pinfo->name_graphic);
  }
  free(pinfo);
}

/***/
int
update_background(void *data, void *user_data) {
  Graphics_drawinfo *gd = (Graphics_drawinfo *) data;
  Graphics *g = (Graphics *) user_data;
  SDL_Rect rect;
  
  if (gd->flags && gd->prev.w > 0) {
    rect = gd->prev;
    rect.x -= g->goffset->x;
    rect.y -= g->goffset->y;
    SDL_BlitSurface(g->background,&rect,g->screen,&(gd->prev));
  }
  return TRUE;
}

int
update_pixmaps(void *data, void *user_data) {
  Graphics_drawinfo *gd = (Graphics_drawinfo *) data;
  Graphics *g = (Graphics *) user_data;

  if (gd->cur.w > 0) {
    SDL_BlitSurface(gd->sprite,&(gd->blitrect),g->screen,&(gd->cur));
    if (gd->flags) {
      graphics_add_rect_update(g,gd->cur);
    }
  } 
  if (gd->prev.w > 0 && (gd->flags&Graphics_POSCHANGED)) {
      graphics_add_rect_update(g,gd->prev);
  }
  return TRUE;
}

int
update_scoreboard(void *data, void *user_data) {
  Graphics_player_info *p = (Graphics_player_info *) data;
  Graphics *g = (Graphics *) user_data;
  SDL_Color black = {0,0,0,0};
  SDL_Rect dstrect;
  int textheight;
  SDL_Surface *text,*surf;
  char str[MAXSTRLEN];
  Vector *vec;
  int anim = 0;

  textheight = TTF_FontHeight(g->font);
  if (p->flags & PINFO_RELOCATED) {
    if (p->name_graphic == NULL) {
      GDEBUG("Rendering text \"%s\" with font pointer %p",
        p->name, g->font);
      p->name_graphic = TTF_RenderText_Blended(g->font,p->name,black);
      assert(p->name_graphic != NULL);
    }
    dstrect.x = p->place->x;
    dstrect.y = p->place->y;
    dstrect.w = p->name_graphic->w;
    dstrect.h = p->name_graphic->h;
    SDL_BlitSurface(p->name_graphic,NULL,g->screen,&dstrect);
    graphics_add_rect_update(g,dstrect);
    /* force the flags to include all others so everything is redone */
    p->flags = 0xffff;
    dstrect.x = p->place->x;
    dstrect.y = p->place->y + p->height - 2;
    dstrect.w = SCOREBOARD_WIDTH;
    dstrect.h = 2;
    SDL_FillRect(g->screen,&dstrect,SDL_MapRGB(g->screen->format,0,0,0));
    graphics_add_rect_update(g,dstrect);
  }
  if (p->flags & PINFO_IMAGE_CHANGED) {
    dstrect.x = p->place->x + SCOREBOARD_WIDTH/2 - p->player_graphic->w/2;
    dstrect.y = p->place->y + textheight;
    dstrect.w = p->player_graphic->w;
    dstrect.h = p->player_graphic->h;
    SDL_FillRect(g->screen,&dstrect,g->scorecolor);
    SDL_BlitSurface(p->player_graphic,NULL,g->screen,&dstrect);
    graphics_add_rect_update(g,dstrect);
  }
  if (p->flags & PINFO_SCORE_CHANGED) {
    snprintf(str,MAXSTRLEN,(_("Score: %6d")),p->score);
    str[MAXSTRLEN - 1] = '\0';
    text  = TTF_RenderText_Blended(g->font,str,black);
    dstrect.x = p->place->x;
    dstrect.y = p->place->y + textheight + p->player_graphic->h;
    dstrect.w = SCOREBOARD_WIDTH;
    dstrect.h = text->h;
    SDL_FillRect(g->screen,&dstrect,g->scorecolor);
    graphics_add_rect_update(g,dstrect);

    dstrect.x = p->place->x + SCOREBOARD_WIDTH - text->w - 2;
    dstrect.w = text->w;
    SDL_BlitSurface(text,NULL,g->screen,&dstrect);
    SDL_FreeSurface(text);
  }
  if (p->flags & PINFO_LIFE_CHANGED) {
    snprintf(str,MAXSTRLEN,"%2d%%", (100 * p->life)/g->maxlife);
    str[MAXSTRLEN-1] = '\0';
    text  = TTF_RenderText_Blended(g->font,str,black);
    dstrect.x = p->place->x;
    dstrect.y = p->place->y + 2 * textheight + p->player_graphic->h;
    dstrect.w = SCOREBOARD_WIDTH;
    dstrect.h = text->h;

    SDL_FillRect(g->screen,&dstrect,g->scorecolor);
    graphics_add_rect_update(g,dstrect);

    dstrect.w = (SCOREBOARD_WIDTH * p->life)/g->maxlife;
    SDL_FillRect(g->screen,&dstrect,
                 p->life < g->maxlife/2 ? 
                   (p->life < g->maxlife/4 ? 
                    SDL_MapRGB(g->screen->format,0xff,0,0) :
                    SDL_MapRGB(g->screen->format,0xff,0xff,0) 
                   )
                 : SDL_MapRGB(g->screen->format,0x0,0xff,0)
                );
    dstrect.x = p->place->x + SCOREBOARD_WIDTH/2 - text->w/2;
    dstrect.w = text->w;
    SDL_BlitSurface(text,NULL,g->screen,&dstrect);
    SDL_FreeSurface(text);
  }
  if (p->flags & PINFO_STACK_CHANGED) {
    snprintf(str,MAXSTRLEN,(_("items: %3d")),p->stacklength);
    str[MAXSTRLEN - 1] = '\0';
    text  = TTF_RenderText_Blended(g->font,str,black);
    dstrect.x = p->place->x;
    dstrect.y = p->place->y + textheight *3 + p->player_graphic->h;
    dstrect.w = SCOREBOARD_WIDTH;
    dstrect.h = MAX(textheight,15);
    SDL_FillRect(g->screen,&dstrect,g->scorecolor);
    graphics_add_rect_update(g,dstrect);

    if (p->stack_top != -1) {
      if (p->foodcaches[p->stack_top] == NULL) {
        vec = new_vector(15,15);
        p->foodcaches[p->stack_top] = 
                  find_scache(g,p->stack_top,vec);
        del_vector(vec);
      }
      surf = graphics_cache_getsprite(p->foodcaches[p->stack_top],g,
                  !p->is_upgraded ? SS_FOOD_NORMAL : SS_FOOD_UPGRADE,
                  &anim);
      dstrect.x = p->place->x + SCOREBOARD_WIDTH/8;
      dstrect.w = 15;
      dstrect.h = 15;
      SDL_BlitSurface(surf,NULL,g->screen,&dstrect);
    }  

    dstrect.x = p->place->x + SCOREBOARD_WIDTH - text->w - 2;
    dstrect.y = dstrect.y + dstrect.h - textheight;
    dstrect.w = text->w;
    dstrect.h = text->h;
    SDL_BlitSurface(text,NULL,g->screen,&dstrect);

    SDL_FreeSurface(text);
  }
  p->flags = 0;
  return TRUE;
}

int
update_player_info(Graphics_player_info *p, Graphics *g, 
                   Model_object *mo,Graphics_object *go) {
  List_ptr *ptr;
  int type;
  int tmp;
  
  if (mo->score != p->score) {
    p->score = mo->score;
    p->flags |= PINFO_SCORE_CHANGED;
  }
  if (mo->life != p->life) {
    p->life = mo->life;
    p->flags |= PINFO_LIFE_CHANGED;
  }
  ptr = new_list_ptr(mo->foodstack);
  if (list_ptr_first(ptr)) {
    type = *((int *)list_ptr_get_data(ptr));
  } else {
    type = -1;
  }
  del_list_ptr(ptr);

  if (type != p->stack_top || p->stacklength != list_length(mo->foodstack)) {
   p->stack_top = type;
   p->stacklength = list_length(mo->foodstack);
   p->flags |= PINFO_STACK_CHANGED;
  }

  if (p->is_upgraded != IS_POWEREDUP(mo)) {
    p->is_upgraded = IS_POWEREDUP(mo);
    p->flags |= PINFO_STACK_CHANGED;
  }
  tmp = 0;
  if (graphics_cache_getsprite(go->scache,g,SS_PLAYER_LSTAND,&tmp) 
      != p->player_graphic) {
    p->player_graphic = 
      graphics_cache_getsprite(go->scache,g,SS_PLAYER_LSTAND,&tmp);
    p->flags |= PINFO_IMAGE_CHANGED;
  }
  return TRUE;
}

int
update_drawing_info(void *data, void *user_data) {
  Model_object *m = (Model_object *) data;
  Graphics *g = (Graphics *) user_data;
  Graphics_object *go;
  SDL_Surface *sprite;
  SDL_Rect rect;

  go = model_object_get_gfx_data(m);
  if (go == NULL && m->gfx_state == GFX_DELETED) {
    return TRUE;
  }

  if (go == NULL) {
    go = new_graphics_object(g,m);
    model_object_set_gfx_data(m,go);
	}

	if (go->pinfo != NULL) {
		update_player_info(go->pinfo,g,m,go);
	}

  if (m->gfx_state == GFX_DELETED) {
    if (go->dinfo->cur.w > 0) {
      rect = go->dinfo->cur;
      rect.x -= g->goffset->x;
      rect.y -= g->goffset->y;
      graphics_add_rect_update(g,go->dinfo->cur);
      SDL_BlitSurface(g->background,&rect,
                      g->screen,&(go->dinfo->cur));
    }
    del_graphics_object(go, g);
    model_object_set_gfx_data(m,NULL);
    return TRUE;
  }

  graphics_drawinfo_set_pos(go->dinfo,g,m->pos,go->size);
  sprite = graphics_cache_getnext_sprite(go->scache,g,go,m);
  graphics_drawinfo_setsprite(go->dinfo,sprite);
  return TRUE;
}

int
free_graphic(void *data, void *user_data) {
  Model_object *mo = (Model_object *) data;
  Graphics *g = (Graphics *) user_data;
  Graphics_object *go;

  go = model_object_get_gfx_data(mo);
  if (go == NULL) 
    return TRUE;
  del_graphics_object(go,g);
  model_object_set_gfx_data(mo,NULL);
  return  TRUE;
}

/*****************************************************************************
 * API Implementation
 *****************************************************************************/

Graphics *
new_graphics(int width, int height, int wwidth, int wheight,
                       int fullscreen) {
	Graphics *g;
	int i;
  int flags = SDL_DOUBLEBUF|SDL_HWPALETTE;
  char fontpath[MAXPATHLEN];

	g = malloc(sizeof(Graphics));
	GDEBUG(_("Setting SDL Videomode: "));
  if (fullscreen == TRUE) {
    flags |= SDL_FULLSCREEN;
  }
	g->screen = SDL_SetVideoMode(width, height, 0, flags);
  if (fullscreen == TRUE) {
    SDL_ShowCursor(SDL_DISABLE);
  }
  assert(width - SCOREBOARD_WIDTH > 40);
  g->gsize = new_vector(width - SCOREBOARD_WIDTH, height);
  g->goffset = new_vector(0,0);
  snprintf(fontpath,MAXPATHLEN,"%s/ffrenzy/fonts/%s",
           DATADIR,FONTNAME);
  fontpath[MAXPATHLEN-1] = '\0';
  g->font = TTF_OpenFont(fontpath,11);
  if (g->font == NULL) {
    WARN(_("Couldn't open default font: %s"),SDL_GetError());
  }


  g->background = SDL_CreateRGBSurface(SDL_SWSURFACE,g->gsize->x,g->gsize->y,
                                    g->screen->format->BitsPerPixel,
                                    g->screen->format->Rmask,
                                    g->screen->format->Gmask,
                                    g->screen->format->Bmask,
                                    g->screen->format->Amask);
  
  if (g->screen->format->palette != NULL) {
    SDL_SetPalette(g->background,SDL_LOGPAL,
                   g->screen->format->palette->colors,0,
                   g->screen->format->palette->ncolors);
  }
  SDL_FillRect(g->background,NULL,
              SDL_MapRGB(g->background->format,0xcc,0xcc,0xff));
  g->scorecolor = SDL_MapRGB(g->screen->format,0x6f,0x6f,0x6f);
  SDL_FillRect(g->screen,NULL,g->scorecolor);
  g->background_changed = TRUE;
	GDEBUG(_("done"));

	g->ratio = new_vector(((float) g->gsize->x / (float) wwidth) * SCALE_X, 
                        ((float) g->gsize->y / (float) wheight) * SCALE_Y);
	SDL_WM_SetCaption("Feeding Frenzy!!!!",NULL);

  for (i = OT_FIRST; i < OT_NRTYPES; i++) {
		GDEBUG("Creating standard types");
    g->sprites[i] = new_list();
    g->svg[i] = NULL;
	}
  for (i = GRAPHICS_LAYER_FIRST; i < GRAPHICS_NRLAYERS; i++) {
    g->layers[i] = new_list();
  }
  g->update_number = 0;
	g->update_queue_len = 0;
  g->update_rects_queue = NULL;

  g->players = new_list();

	return g;
}

void del_graphics(Graphics * g, Model * m) {
  int i;
  Graphics_cache *c;
  for (i = 0; i < OT_NRTYPES; i++) {
    if (g->svg[i] != NULL) del_sprite_source(g->svg[i]);
    while ((c = list_pop(g->sprites[i])) != NULL) {
      del_graphics_cache(c);
    }
    del_list(g->sprites[i]);
	}
	GDEBUG("Freeing all pixmaps");
	model_foreach_object(m, free_graphic, g);
	GDEBUG("Freeing main surface");
	SDL_FreeSurface(g->screen);
	GDEBUG("Freeing graphics storage");
  del_vector(g->gsize);
  del_vector(g->goffset);
  del_vector(g->ratio);
  for (i = 0; i < GRAPHICS_NRLAYERS; i++) {
    del_list(g->layers[i]);
  }
  free(g->update_rects_queue);
  del_list(g->players);
  if (g->font != NULL) {
    TTF_CloseFont(g->font);
  }
	free(g);
}

int 
graphics_update(Graphics * g, Model * m) {
  int i;
  SDL_Rect rect;

  g->update_number = 0;
  if (g->background_changed == TRUE) {
    rect.x = g->goffset->x;
    rect.y = g->goffset->y;
    rect.w = g->gsize->x;
    rect.h = g->gsize->y;
    SDL_BlitSurface(g->background,NULL,g->screen,&rect);
  }
	model_foreach_object (m, update_drawing_info, g);
  for (i = GRAPHICS_LAYER_FIRST; i < GRAPHICS_NRLAYERS; i++) {
	  list_foreach(g->layers[i], update_background, g);
  }
  for (i = GRAPHICS_LAYER_FIRST; i < GRAPHICS_NRLAYERS; i++) {
	  list_foreach(g->layers[i], update_pixmaps, g);
  }

  g->maxlife = m->settings->player_max_life;
  list_foreach(g->players,update_scoreboard,g);

  if (g->background_changed == FALSE) {
    SDL_UpdateRects(g->screen,g->update_number,g->update_rects_queue);
  } else {
    SDL_Flip(g->screen);
    g->background_changed = FALSE;
  }

	return TRUE;
}

static int
graphics_compare_score(void *data0, void *data1,void *user_data) {
  Graphics_player_info *pinfo0 = (Graphics_player_info *) data0;
  Graphics_player_info *pinfo1 = (Graphics_player_info *) data1;
  if (pinfo0->life == 0 && pinfo1->life > 0) {
    return 1;
  } else if (pinfo0->life > 0 && pinfo1->life == 0) {
    return -1;
  } else {
    return pinfo1->score - pinfo0->score;
  }
}

int
graphics_showscores(Graphics *g) {
  char fontpath[MAXPATHLEN];
  char string[MAXSTRLEN];
  TTF_Font *font;
  List_ptr *ptr;
  SDL_Rect rect;
  Graphics_player_info *pinfo;
  SDL_Color black = {0,0,0,0};
  SDL_Color fontback = {0xcc,0xcc,0xcc,0};
  SDL_Surface *text;

  list_sort(g->players,graphics_compare_score,NULL);

  ptr = new_list_ptr(g->players);
  g->update_number = 0;
  if (!list_ptr_first(ptr)) {
    del_list_ptr(ptr);
    return FALSE;
  }
  snprintf(fontpath,MAXPATHLEN,"%s/ffrenzy/fonts/%s",
           DATADIR,FONTNAME);
  fontpath[MAXPATHLEN-1] = '\0';
  font = TTF_OpenFont(fontpath,g->gsize->y/15);
  rect.y = g->goffset->y + g->gsize->y/4;
  do {
    assert((pinfo = list_ptr_get_data(ptr)) != NULL);

    snprintf(string,MAXSTRLEN,"%c %s: %d  ", pinfo->life > 0 ? ' ' : 'X',
                                           pinfo->name,pinfo->score);
    string[MAXSTRLEN-1] = '\0';
    text = TTF_RenderText_Shaded(font,string,black,fontback);
    /* Either SDL_ALPHA_TRANSPARENT of SDL_ALPHA_OPAQUE is 0,
     * dependent on the version of SDL */
    if (g->screen->format->BytesPerPixel != 1) {
      SDL_SetAlpha(text,SDL_SRCALPHA,
                 SDL_ALPHA_TRANSPARENT - SDL_ALPHA_TRANSPARENT/2 
                 + SDL_ALPHA_OPAQUE/2);
    }
    rect.x = g->gsize->x/2 - text->w/2 + g->goffset->x;
    rect.w = text->w;
    rect.h = text->h;
    SDL_BlitSurface(text,NULL,g->screen,&rect);
    graphics_add_rect_update(g,rect);
    rect.y += text->h + 10; 
    SDL_FreeSurface(text);
  } while (list_ptr_next(ptr) == TRUE);

  SDL_UpdateRects(g->screen,g->update_number,g->update_rects_queue);
  TTF_CloseFont(font);
  return TRUE;
}

int graphics_add_object(Graphics * g, Model_object *mo, char *SVGserie) {
  Graphics_svg_series *gss;
  Graphics_sprite_source *spritesource;
  Graphics_object *go;

	GDEBUG("Adding object");
  gss = parse_svg_series(SVGserie);
  if (gss == NULL) { 
    WARN(_("Couldn't parse SVGserie for object"));
    return FALSE;
  }
  spritesource = gss_to_sprite_source(gss);
	del_gss(gss);
  go = model_object_get_gfx_data(mo);
  if (go == NULL) {
    go = new_graphics_object(g,mo);
  }

  if (go->private_graphic) {
    del_sprite_source(go->scache->svg);
    graphics_cache_change_svg(go->scache,spritesource);
  } else {
    go->scache = new_graphics_cache(mo->type,go->size,spritesource);
  }
  model_object_set_gfx_data(mo,go);
  return TRUE;
}

int 
do_change_cache_svg(void *data, void *user_data) {
  Graphics_cache *c = (Graphics_cache *) data;
  Graphics_sprite_source *s = (Graphics_sprite_source *) user_data;
  graphics_cache_change_svg(c,s);
  return TRUE;
}

int 
graphics_register_type(Graphics * g, Object_type type, char *SVGserie) {
  Graphics_svg_series *gss;

  gss = parse_svg_series(SVGserie);
  if (gss == NULL) { 
    WARN(_("Couldn't parse SVGserie for type: %d"),type);
    return FALSE;
  }
  if (g->svg[type] != NULL) {
    del_sprite_source(g->svg[type]);
  }
  g->svg[type] = gss_to_sprite_source(gss);
  list_foreach(g->sprites[type],do_change_cache_svg,g->svg[type]);
  del_gss(gss);
  return TRUE;
}

int
graphics_set_background(Graphics *g, char *SVGserie) {
  Graphics_svg_series *gss;
  Graphics_sprite_anim *anim;
  Graphics_sprite_source *sprites;

  gss = parse_svg_series(SVGserie);
  if (gss == NULL) {
    WARN(_("Couldn't parse SVGserie for background"));
    return FALSE;
  }
  sprites = gss_to_sprite_source(gss);
  anim = draw_sprite_anim(g->background,g->gsize,SS_SCENERY_NORMAL,sprites);
  SDL_FillRect(g->background,NULL,
               SDL_MapRGB(g->background->format,0xcc,0xcc,0xff));
  SDL_BlitSurface(anim->anim[0],NULL,g->background,NULL);
  del_graphics_sprite_anim(anim);
  del_sprite_source(sprites);
  g->background_changed = TRUE;
  return TRUE;
}

int
pinfo_updatelocation(void *data, void *user_data) {
  Graphics_player_info *p = (Graphics_player_info *) data;
 Vector *vec = (Vector *) user_data;

  p->place->x = vec->x;
  p->place->y = vec->y;
  vec->y += p->height;
  p->flags |= PINFO_RELOCATED;

  return TRUE;
}

int 
graphics_add_player(Graphics * g, Model_object *obj, char *name,int is_local) {
	Graphics_object* go = (Graphics_object*) model_object_get_gfx_data(obj);
  Vector *vec; 

	if (go == NULL) {
		go = new_graphics_object(g,obj);
	  model_object_set_gfx_data(obj, go);
	}
	go->pinfo = new_graphics_player_info(name, is_local);
  go->pinfo->height = go->size->y + 3* TTF_FontHeight(g->font) 
                      + MAX(15,TTF_FontHeight(g->font) + 5);

  if (is_local) {
    list_prepend(g->players,go->pinfo);
  } else {
    list_append(g->players,go->pinfo);
  }

  vec = new_vector(g->screen->w - SCOREBOARD_WIDTH,0);
  list_foreach(g->players,pinfo_updatelocation,vec);
  del_vector(vec);
	return TRUE;
}
